/*******************************************************************************/
/* Permission is hereby granted, free of charge, to any person or organization */
/* obtaining a copy of the software and accompanying documentation covered by  */
/* this license (the "Software") to use, reproduce, display, distribute,       */
/* execute, and transmit the Software, and to prepare derivative works of the  */
/* Software, and to permit third-parties to whom the Software is furnished to  */
/* do so, all subject to the following:                                        */
/*                                                                             */
/* The copyright notices in the Software and this entire statement, including  */
/* the above license grant, this restriction and the following disclaimer,     */
/* must be included in all copies of the Software, in whole or in part, and    */
/* all derivative works of the Software, unless such copies or derivative      */
/* works are solely in the form of machine-executable object code generated by */
/* a source language processor.                                                */
/*                                                                             */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  */
/* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    */
/* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT   */
/* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE   */
/* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, */
/* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER */
/* DEALINGS IN THE SOFTWARE.                                                   */
/*******************************************************************************/

#include <lfp/socket.h>
#include <lfp/fcntl.h>
#include <lfp/unistd.h>

#if defined(HAVE_GETPEERUCRED)
#include <ucred.h>
#endif

DSO_PUBLIC int
lfp_socket(int domain, int type, int protocol, uint64_t flags)
{
#if defined(SOCK_CLOEXEC) && defined(SOCK_NONBLOCK)
    return socket(domain,
                  type | ((flags & O_CLOEXEC)  ? SOCK_CLOEXEC  : 0) \
                       | ((flags & O_NONBLOCK) ? SOCK_NONBLOCK : 0),
                  protocol);
#else
    int fd = socket(domain, type, protocol);
    if (fd < 0) { goto error_return; }

    if (((flags & O_CLOEXEC)  && (lfp_set_fd_cloexec(fd, true)  < 0)) || \
        ((flags & O_NONBLOCK) && (lfp_set_fd_nonblock(fd, true) < 0))) {
        goto error_close;
    }

    return fd;

  error_close:
    close(fd);
  error_return:
    return -1;
#endif
}

DSO_PUBLIC int
lfp_accept(int             sockfd,
           struct sockaddr *addr,
           socklen_t       *addrlen,
           uint64_t        flags)
{
// FIXME: This branch requires that SOCK_CLOEXEC and SOCK_NONBLOCK be
// defined when accept4(2) is present. Should we add an Autoconf check ?
#if defined(HAVE_ACCEPT4) && defined(SOCK_CLOEXEC) && defined(SOCK_NONBLOCK)
    return accept4(sockfd, addr, addrlen,
                   ((flags & O_CLOEXEC)  ? SOCK_CLOEXEC  : 0) | \
                   ((flags & O_NONBLOCK) ? SOCK_NONBLOCK : 0));
#else
    int fd = accept(sockfd, addr, addrlen);
    if (fd < 0) { goto error_return; }

    if (((flags & O_CLOEXEC)  && (lfp_set_fd_cloexec(fd, true)  < 0)) || \
        ((flags & O_NONBLOCK) && (lfp_set_fd_nonblock(fd, true) < 0))) {
        goto error_close;
    }

    return fd;

  error_close:
    close(fd);
  error_return:
    return -1;
#endif
}


DSO_PUBLIC struct cmsghdr*
lfp_cmsg_firsthdr (struct msghdr* msgh)
{
  return CMSG_FIRSTHDR(msgh);
}

DSO_PUBLIC struct cmsghdr*
lfp_cmsg_nxthdr (struct msghdr* msgh, struct cmsghdr* cmsg)
{
  return CMSG_NXTHDR(msgh, cmsg);
}

DSO_PUBLIC size_t
lfp_cmsg_space (size_t length)
{
  return CMSG_SPACE(length);
}

DSO_PUBLIC size_t
lfp_cmsg_len (size_t length)
{
  return CMSG_LEN(length);
}

DSO_PUBLIC void*
lfp_cmsg_data (struct cmsghdr* cmsg)
{
  return CMSG_DATA(cmsg);
}


DSO_PUBLIC int
lfp_getpeereid(int sockfd, uid_t *euid, gid_t *egid)
{
    SYSCHECK(EFAULT, euid == NULL || egid == NULL);
#if defined(HAVE_GETPEEREID)
    return getpeereid(sockfd, euid, egid);
#elif defined(HAVE_GETPEERUCRED)
    ucred_t **ucred = NULL;

    SYSGUARD(getpeerucred(sockfd, ucred));
    *euid = ucred_geteuid(*ucred);
    *egid = ucred_getegid(*ucred);
    ucred_free(*ucred);

    return (*euid < 0 || *egid < 0) ? -1 : 0;
#elif defined(SO_PEERCRED)
    struct ucred ucred;
    socklen_t len = sizeof(ucred);

    SYSGUARD(getsockopt(sockfd, SOL_SOCKET, SO_PEERCRED, &ucred, &len));

    *euid = ucred.uid;
    *egid = ucred.gid;

    return 0;
#endif
}
