/*
 *    con_i18n.cpp
 *
 *    Copyright (c) 1998, Zdenek Kabelac
 *
 *    I18N support by kabi@fi.muni.cz
 *
 *    written as plain 'C' module and might be used
 *    in other programs to implement I18N support
 */


#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>
#include <ctype.h>
#if defined(AIX)
#include <sys/select.h>
#endif
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include "con_i18n.h"

#define KEYMASK 0xff
#define KEYBYTEMAX 0xf00

#ifdef USE_HARD_REMAP

/*
 * This part is used when your Xserver doesn't work well with XKB extension
 */


/* Keyboard definition file - currently only Czech national keyboard
 * write your own keyboard for your language
 * And remember - this is supposed to be used only if your Xserver
 * is not supporting keyboard extension
 */
#include "con_remap.h"


static struct keyboardRec *I18NFindKeyboard(long i, long remap)
{
    static struct keyboardRec **kMap = NULL;
    struct keyboardRec *kbdActual;

    if (kMap == NULL) {
	char *r;
	int i;

	/* initialize used keyboards */
	r = getenv("_LIBI18NFORCE");
	if ((r == NULL) || (strlen(r) < 1)) {
	    r = getenv("LC_CTYPE");
	    if ((r == NULL) || (strlen(r) < 1)) {
		r = getenv("LANG");
		if ((r == NULL) || (strlen(r) < 1))
		    r = "";
	    }
	}
	for (i = 0; ; i++) {
	    kMap = nationalKey[i].kbd;
	    if ((nationalKey[i].lang == NULL)
		|| strstr(r, nationalKey[i].lang))
		break;
	}
    }

    /* Select keyboard map - for now only the first three keyboards */
    if (!i)
	kbdActual = kMap[0];
    else if (!remap)
	kbdActual = kMap[1];
    else
	kbdActual = kMap[2];

    return kbdActual;
}


/*
 * Three keyboards are supported for each language setting
 * - there is a simple switch mechanism
 * If you think its not enough - tell me how many
 * keyboard would you need and how would you expect to
 * change between them
 */

/*
 * Quite a complex function to convert normal keys
 * to dead-keys and remapped keys
 *
 * this function will not work in the thread environment
 * but I suppose no one expect this :-)
 */
int I18NKeyAnalyze(XKeyEvent * event, char *buffer, /*FOLD00*/
		   int nbytes, KeySym * keysym)
{
    static KeySym keysymL;
    static int nbytesL;
    static long prev_state = 0, local_switch = 0, keypos = 0,
    last_keypos = 0, remap = 0;
    static Time time = 0;
    long i;
    struct keyboardRec *kbdActual;


    if (time == event->time) {
	/* return already converted values for this event */
	/* suppose no one writes 1000 chars per second */
	*keysym = keysymL;
	buffer[0] = keysymL & KEYMASK;
#if 0
	fprintf(stderr, "RETURN: 0x%-7lx%-20s%d  %c  %p %d %d\n ", *keysym,
		XKeysymToString(*keysym), nbytesL, buffer[0],
		buffer, event, event->send_event);
#endif
	return nbytesL;
    }

    time = event->time;			/* new time */
    keysymL = *keysym;
    nbytesL = nbytes;


    /* Find the state of keyboard
     * Check for different ISO group or modifier 5
     * So to activate remaping, you need to press at least
     * ScrollLock which is usually mapped as modifier 5
     * But ISO_Next_Group works as well
     */
    i = ((event->state | local_switch) & 0xFFA0);

    if (i != prev_state) {
	/* reset table position */
	last_keypos = keypos = 0;
	prev_state = i;
    }

    if (event->type == KeyPress) {
	int r = 0;
	if (i && ((*keysym == XK_Pause) || (*keysym == XK_F21))) {
	    remap++;
	    if (remap > 1)
		remap = 0; /* ok this could be one way to
		support big number of various keyboards */
	    /*fprintf(stderr, "MODIFIER:  %ld  %ld\n",
		    event->send_event, remap);*/
	    r++;
	} else if (*keysym == XK_F23) /* for SunOS use extra switch */ {
	    local_switch ^= (1UL<< 12);
            r++;
	}
	if (r) {
	    keysymL = nbytesL = buffer[0] = *keysym = 0;
	    return 0;
	}
    }
    /*
     * Check for already remapped ISO8859-X keys
     * this should not normaly happen :-)
     * as we need to use this hack
     */

    if (((*keysym > KEYMASK) && (*keysym < KEYBYTEMAX))
	|| (*keysym > 0xffffff00)) {
	keysymL = *keysym &= KEYMASK;
	buffer[0] = (char) *keysym;
	nbytesL = 1;
	return 1;
    }

    kbdActual = I18NFindKeyboard(i, remap);

    if (event->type == KeyPress) {
	long i = last_keypos;

	/*
	 * Search for DeadKey -> do loop over all tables.
	 *
	 * Note: We can define ONE DeadKey and use
	 * it for several tables sequentially
	 */
	for (;;) {
	    i++;
	    if ((kbdActual[0].tab == NULL) || kbdActual[i].tab == NULL) {
		i = 0;
		if (kbdActual[i].tab == NULL) {
		    /* Looks like empty table -> IGNORE */
		    keypos = i;
		    return nbytes;
		}
	    }
	    if (i == last_keypos)
		break;

	    if (kbdActual[i].deadkey == *keysym) {
		keypos = kbdActual[i].next;
		/* Found DeadKey -> delete it
		 * and mark actual position for next search */
		last_keypos = i;
		event->type = 0;
		keysymL = buffer[0] = *keysym = 0;
                nbytesL = 0;
		return 0;
	    }
	}
    } else if (keypos)
	return 0;		/* ignore key release */

    /* Now we know it is not a DeadKey and we
     * are in selected remaping keyboard table */
    /*
    printf("** key:%5ld\tstatus:0x%lx     keypos: %ld  %ld\n",
    *keysym, prev_state, keypos, kbdActual[keypos].tab[i].key_english);
    */
    if (*keysym < KEYBYTEMAX) {
	/*
	 * this is selected constant and will change when
	 * this editor will be able to write Japaniese :-)
	 */
	int i = 0;

	/* remaping only real keys */
	while (kbdActual[keypos].tab[i].key_english != 0) {
	    if (*keysym == kbdActual[keypos].tab[i].key_english) {
		*keysym = kbdActual[keypos].tab[i].key_remap;
		break;
	    }
	    i++;
	}
	last_keypos = keypos = kbdActual[keypos].next;
#if 0
	fprintf(stderr, "** remap: %3ld %3ld\n", keypos, *keysym);
#endif
	keysymL = (unsigned short) *keysym;
	buffer[0] = *keysym & KEYMASK;
	nbytesL = 1;
	return 1;
    }
    return 0;
}
 /*FOLD00*/
#else

/*********************************************
 *                                           *
 *   Standart methods for reading Keyboard   *
 *                                           *
 *********************************************/

/* ISO-8859-2 key-change

 * All these functions are for keyboard reading.
 * Correct displaing of this text is another thing,
 * but as I only need ISO-8859 encoding support,
 * I don't care about this (for now).
 */
static int I18NKeyAnalyze(XKeyEvent * event, char *buffer, /*fold00*/
			  int nbytes, KeySym * keysym)
{
    KeySym t = (unsigned char) buffer[0];

    /*
     * ISO-8859-2 is using some characters from 8859-1 and
     * rest of them is located between 0x100 - 0x200 in 'X' so
     * with ISO-8859-2 font we'll remap them down bellow < 0x100
     * This is mostly true for all Latin-X alphas, just
     * special font to display them correctly is needed.
     * This jobs does Xserver - and keysymbol is returned
     * in the 1st. byte of keyName string.
     */
    if ((nbytes == 1) && *keysym < KEYBYTEMAX)
	*keysym = t;
#ifdef USE_HACK_FOR_BAD_XSERVER
    /*
     * this is really hack - but some Xservers are somewhat
     * strange, so we remap character manually
     */
    else if (!nbytes && (*keysym > KEYMASK) && (*keysym < KEYBYTEMAX)) {
	*keysym &= KEYMASK;
	buffer[0] = *keysym & KEYMASK;
	nbytes = 1;
    }
#endif
    return nbytes;
}
 /*FOLD00*/
#endif

/*
 * Initialize I18N functions - took me hours to
 * figure out how this works even though it was
 * cut & pasted from 'xterm' sources, but as 'xterm'
 * is using Xt Toolkit some things have to be made
 * different
 */
XIC I18NInit(Display * display, Window win, unsigned long *mask) /*FOLD00*/
{
    XIC xic = (XIC) NULL;

#if XlibSpecificationRelease >= 6
    XIM xim = (XIM) NULL;
    XIMStyles *xim_styles;
    XIMStyle input_style = 0;
    char *s, tmp[256];
    int found = False;

    *mask = 0;

    /* Locale setting taken from XtSetLanguageProc */
    if (!(s = setlocale(LC_ALL, "")))
	fprintf(stderr, "I18N: Locale not supported by C library, "
		"locale unchanged!\n");

    if (!XSupportsLocale()) {
	fprintf(stderr, "I18N: Locale not supported by Xlib, "
		"locale set to C!\n");
	s = setlocale(LC_ALL, "C");
    }
    if (!XSetLocaleModifiers(""))
	fprintf(stderr, "I18N: X locale modifiers not supported, "
		"using default\n");

    if ((xim = XOpenIM(display, NULL, NULL, NULL)) == NULL) {
	fprintf(stderr, "I18N: Failed to open input method\n");
	return xic;
    }
    if (XGetIMValues(xim, XNQueryInputStyle, &xim_styles, NULL)
	|| (xim_styles == NULL)) {
	fprintf(stderr, "I18N: Input method doesn't support any style\n");
	XCloseIM(xim);
	return xic;
    }

    /*
     * This is some kind of debugging message to inform user
     * that something is wrong with his system
     */
    if ((s != NULL) && (strstr(s, XLocaleOfIM(xim)) == NULL)) {
	fprintf(stderr, "I18N Warning: System locale: \"%s\" differs from "
		"IM locale: \"%s\"...\n", s, XLocaleOfIM(xim));
#ifdef USE_HARD_REMAP
	fprintf(stderr, "I18N: hack should still work...\n");
#endif
    }

    /*
     * This part is cut&paste from other sources
     * There is no reason to do it this way, because
     * the only input style currently supported is Root
     * but for the future extension I'll leave it here
     */

    strcpy(tmp, XIM_INPUT_STYLE);
    for (s = tmp; s && !found;) {
	char *ns, *end;
	int i;

	while ((*s != 0) && isspace(*s))
	    s++;

	if (*s == 0)
	    break;

	if ((ns = end = strchr(s, ',')) != 0)
	    ns++;
	else
	    end = s + strlen(s);

	while (isspace(*end))
	    end--;
	*end = '\0';

	if (!strcmp(s, "OverTheSpot"))
	    input_style = (XIMPreeditPosition | XIMStatusArea);
	else if (!strcmp(s, "OffTheSpot"))
	    input_style = (XIMPreeditArea | XIMStatusArea);
	else if (!strcmp(s, "Root"))
	    input_style = (XIMPreeditNothing | XIMStatusNothing);

	for (i = 0; (unsigned short) i < xim_styles->count_styles; i++)
	    if (input_style == xim_styles->supported_styles[i]) {
		found = True;
		break;
	    }
	s = ns;
    }
    XFree(xim_styles);

    if (!found) {
	fprintf(stderr, "I18N: Input method doesn't support my preedit type\n");
	XCloseIM(xim);
	return xic;
    }
    /* This program only understand the Root preedit_style yet */
    if (input_style != (XIMPreeditNothing | XIMStatusNothing)) {
	fprintf(stderr, "I18N: This program only supports the "
		"'Root' preedit type\n");
	XCloseIM(xim);
	return xic;
    }
    xic = XCreateIC(xim, XNInputStyle, input_style,
		    XNClientWindow, win,
		    XNFocusWindow, win,
		    NULL);
    if (xic == NULL) {
	fprintf(stderr, "I18N: Failed to create input context\n");
	XCloseIM(xim);
    } else if (XGetICValues(xic, XNFilterEvents, mask, NULL))
	fprintf(stderr, "I18N: Can't get Event Mask\n");
#else
    *mask = 0;
#endif
    return xic;
}
 /*FOLD00*/

void I18NFocusIn(XIC xic) /*fold00*/
{
#if XlibSpecificationRelease >= 6
    if (xic != NULL)
	XSetICFocus(xic);
#endif
}
 /*FOLD00*/

void I18NFocusOut(XIC xic) /*fold00*/
{
#if XlibSpecificationRelease >= 6
    if (xic != NULL)
	XUnsetICFocus(xic);
#endif
}
 /*FOLD00*/

/*
 * Lookup correct keysymbol from keymap event
 */
int I18NLookupString(XKeyEvent * event, char *buffer, int nbytes, /*FOLD00*/
		     KeySym * keysym, XIC xic)
{
    static int showKeys = 0;

#if XlibSpecificationRelease >= 6
    if (xic != NULL) {
	if (event->type == KeyPress) {
	    Status status_return;

	    /* No KeyRelease events here ! */
	    nbytes = XmbLookupString(xic, event, buffer, nbytes,
				     keysym, &status_return);
	}
    } else
#endif
    do {
	static XComposeStatus compose_status = { NULL, 0 };
	nbytes = XLookupString(event, buffer, nbytes,
			       keysym, &compose_status);
    } while (0);


    if (showKeys) {
	fprintf(stderr, "Key: 0x%04lx  '%s'\tKeyEventState:0x%x\t",
		*keysym, XKeysymToString(*keysym), event->state);
	if (nbytes && isprint(buffer[0])) {
	    buffer[nbytes] = 0;
	    fprintf(stderr, "String:'%s' Size:%2d  ", buffer, nbytes);
	}
	fputs("\n", stderr);
    }
    if (((*keysym == XK_F1) || (*keysym == XK_F11))
	&& ((event->state & (ShiftMask | ControlMask))
	    == (ShiftMask | ControlMask))
	&& (event->type == KeyPress))
	showKeys = !showKeys;

    return I18NKeyAnalyze(event, buffer, nbytes, keysym);
}
 /*FOLD00*/
