; Tests of the SHA-512 formalization
;
; Copyright (C) 2008-2011 Eric Smith and Stanford University
; Copyright (C) 2013-2019 Kestrel Institute
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SHA2")

(include-book "sha-512")
(include-book "../../bv-lists/string-to-bits")
(include-book "../../utilities/strings/hexstrings")
(include-book "../../../std/testing/assert-equal")

;; Test of the padding operation (from Section 5.1.2):
(acl2::assert-equal
 (sha-512-pad-message '(0 1 1 0 0 0 0 1   0 1 1 0 0 0 1 0   0 1 1 0 0 0 1 1))
 (append '(0 1 1 0 0 0 0 1   0 1 1 0 0 0 1 0   0 1 1 0 0 0 1 1)
         (list 1)
         (repeat 871 0)
         (repeat 123 0) '(1 1 0 0 0)))

;; Tests from  https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512.pdf:

(acl2::assert-equal (sha-512 (acl2::string-to-bits "abc"))
                    (acl2::hexstring=>ubyte8s "DDAF35A193617ABACC417349AE20413112E6FA4E89A97EA20A9EEEE64B55D39A2192992A274FC1A836BA3C23A3FEEBBD454D4423643CE80E2A9AC94FA54CA49F"))

(acl2::assert-equal (sha-512 (acl2::string-to-bits "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"))
                    (acl2::hexstring=>ubyte8s "8E959B75DAE313DA8CF4F72814FC143F8F7779C6EB9F7FA17299AEADB6889018501D289E4900F7E4331B99DEC4B5433AC7D329EEB6DD26545E96E55B874BE909"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s ""))
                    (acl2::hexstring=>ubyte8s "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "21"))
                    (acl2::hexstring=>ubyte8s "3831a6a6155e509dee59a7f451eb35324d8f8f2df6e3708894740f98fdee23889f4de5adb0c5010dfb555cda77c8ab5dc902094c52de3278f35a75ebc25f093a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9083"))
                    (acl2::hexstring=>ubyte8s "55586ebba48768aeb323655ab6f4298fc9f670964fc2e5f2731e34dfa4b0c09e6e1e12e3d7286b3145c61c2047fb1a2a1297f36da64160b31fa4c8c2cddd2fb4"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "0a55db"))
                    (acl2::hexstring=>ubyte8s "7952585e5330cb247d72bae696fc8a6b0f7d0804577e347d99bc1b11e52f384985a428449382306a89261ae143c2f3fb613804ab20b42dc097e5bf4a96ef919b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "23be86d5"))
                    (acl2::hexstring=>ubyte8s "76d42c8eadea35a69990c63a762f330614a4699977f058adb988f406fb0be8f2ea3dce3a2bbd1d827b70b9b299ae6f9e5058ee97b50bd4922d6d37ddc761f8eb"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "eb0ca946c1"))
                    (acl2::hexstring=>ubyte8s "d39ecedfe6e705a821aee4f58bfc489c3d9433eb4ac1b03a97e321a2586b40dd0522f40fa5aef36afff591a78c916bfc6d1ca515c4983dd8695b1ec7951d723e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "38667f39277b"))
                    (acl2::hexstring=>ubyte8s "85708b8ff05d974d6af0801c152b95f5fa5c06af9a35230c5bea2752f031f9bd84bd844717b3add308a70dc777f90813c20b47b16385664eefc88449f04f2131"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "b39f71aaa8a108"))
                    (acl2::hexstring=>ubyte8s "258b8efa05b4a06b1e63c7a3f925c5ef11fa03e3d47d631bf4d474983783d8c0b09449009e842fc9fa15de586c67cf8955a17d790b20f41dadf67ee8cdcdfce6"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "6f8d58b7cab1888c"))
                    (acl2::hexstring=>ubyte8s "a3941def2803c8dfc08f20c06ba7e9a332ae0c67e47ae57365c243ef40059b11be22c91da6a80c2cff0742a8f4bcd941bdee0b861ec872b215433ce8dcf3c031"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "162b0cf9b3750f9438"))
                    (acl2::hexstring=>ubyte8s "ade217305dc34392aa4b8e57f64f5a3afdd27f1fa969a9a2608353f82b95cfb4ae84598d01575a578a1068a59b34b5045ff6d5299c5cb7ee17180701b2d1d695"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "bad7c618f45be207975e"))
                    (acl2::hexstring=>ubyte8s "5886828959d1f82254068be0bd14b6a88f59f534061fb20376a0541052dd3635edf3c6f0ca3d08775e13525df9333a2113c0b2af76515887529910b6c793c8a5"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "6213e10a4420e0d9b77037"))
                    (acl2::hexstring=>ubyte8s "9982dc2a04dff165567f276fd463efef2b369fa2fbca8cee31ce0de8a79a2eb0b53e437f7d9d1f41c71d725cabb949b513075bad1740c9eefbf6a5c6633400c7"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "6332c3c2a0a625a61df71858"))
                    (acl2::hexstring=>ubyte8s "9d60375d9858d9f2416fb86fa0a2189ee4213e8710314fd1ebed0fd158b043e6e7c9a76d62c6ba1e1d411a730902309ec676dd491433c6ef66c8f116233d6ce7"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f47be3a2b019d1beededf5b80c"))
                    (acl2::hexstring=>ubyte8s "b94292625caa28c7be24a0997eb7328062a76d9b529c0f1d568f850df6d569b5e84df07e9e246be232033ffac3adf2d18f92ab9dacfc0ecf08aff7145f0b833b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "b1715f782ff02c6b88937f054116"))
                    (acl2::hexstring=>ubyte8s "ee1a56ee78182ec41d2c3ab33d4c41871d437c5c1ca060ee9e219cb83689b4e5a4174dfdab5d1d1096a31a7c8d3abda75c1b5e6da97e1814901c505b0bc07f25"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9bcd5262868cd9c8a96c9e82987f03"))
                    (acl2::hexstring=>ubyte8s "2e07662a001b9755ae922c8e8a95756db5341dc0f2e62ae1cf827038f33ce055f63ad5c00b65391428434ddc01e5535e7fecbf53db66d93099b8e0b7e44e4b25"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "cd67bd4054aaa3baa0db178ce232fd5a"))
                    (acl2::hexstring=>ubyte8s "0d8521f8f2f3900332d1a1a55c60ba81d04d28dfe8c504b6328ae787925fe0188f2ba91c3a9f0c1653c4bf0ada356455ea36fd31f8e73e3951cad4ebba8c6e04"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "6ba004fd176791efb381b862e298c67b08"))
                    (acl2::hexstring=>ubyte8s "112e19144a9c51a223a002b977459920e38afd4ca610bd1c532349e9fa7c0d503215c01ad70e1b2ac5133cf2d10c9e8c1a4c9405f291da2dc45f706761c5e8fe"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c6a170936568651020edfe15df8012acda8d"))
                    (acl2::hexstring=>ubyte8s "c36c100cdb6c8c45b072f18256d63a66c9843acb4d07de62e0600711d4fbe64c8cf314ec3457c90308147cb7ac7e4d073ba10f0ced78ea724a474b32dae71231"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "61be0c9f5cf62745c7da47c104597194db245c"))
                    (acl2::hexstring=>ubyte8s "b379249a3ca5f14c29456710114ba6f6136b34c3fc9f6fb91b59d491af782d6b237eb71aaffdd38079461cf690a46d9a4ddd602d19808ab6235d1d8aa01e8200"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "e07056d4f7277bc548099577720a581eec94141d"))
                    (acl2::hexstring=>ubyte8s "59f1856303ff165e2ab5683dddeb6e8ad81f15bb578579b999eb5746680f22cfec6dba741e591ca4d9e53904837701b374be74bbc0847a92179ac2b67496d807"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "67ebda0a3573a9a58751d4169e10c7e8663febb3a8"))
                    (acl2::hexstring=>ubyte8s "13963f81cfabfca71de4739fd24a10ce3897bba1d716907fc0a28490c192a7fc3ccb8db1f91af7a2d250d6617f0dfd1519d221d618a02e3e3fa9041cf35ed1ea"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "63e09db99eb4cd6238677859a567df313c8520d845b4"))
                    (acl2::hexstring=>ubyte8s "9083e5348b08eb9810b2d15781d8265845410de54fe61750d4b93853690649adc6e72490bc2b7c365e2390573d9414becc0939719e0cb78eca6b2c80c2fda920"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f3e06b4bd79e380a65cb679a98ccd732563cc5ebe892e2"))
                    (acl2::hexstring=>ubyte8s "6b315f106b07c59eedc5ab1df813b3c0b903060e7217cc010e9070278512a885008dac8b2472a521e77835a7f4deadc1d591aa23b624b69948a99bb60121c54e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "16b17074d3e3d97557f9ed77d920b4b1bff4e845b345a922"))
                    (acl2::hexstring=>ubyte8s "6884134582a760046433abcbd53db8ff1a89995862f305b887020f6da6c7b903a314721e972bf438483f452a8b09596298a576c903c91df4a414c7bd20fd1d07"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3edf93251349d22806bed25345fd5c190aac96d6cdb2d758b8"))
                    (acl2::hexstring=>ubyte8s "299e0daf6605e5b0c30e1ec8bb98e7a3bd7b33b388bdb457452dab509594406c8e7b841e6f4e75c8d6fbd614d5eb9e56c359bfafb4285754787ab72b46dd33f0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "b2d5a14f01e6b778888c562a059ec819ad89992d16a09f7a54b4"))
                    (acl2::hexstring=>ubyte8s "ab2e7d745d8ad393439af2a3fbc9cdc25510d4a04e78b526e12b1c0be3b22966872ebe652e2f46ed5c5acecd2f233a9175dd295ebeb3a0706fc66fa1b137042b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "844b66f12ba0c5f9e92731f571539d1eef332e1549a49dbfa4c6de"))
                    (acl2::hexstring=>ubyte8s "c3f9c5781925774783ae9d839772d7513dfcea8c5af8da262c196f9fe80135b2b0c8c6ca0a1604e0a3460247620de20b299f2db7871982d27c2176ae5fa7ad65"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "6b6cc692d39860b1f30203653e25d09c01e6a8043c1a9cb8b249a41e"))
                    (acl2::hexstring=>ubyte8s "2e5263d9a4f21b210e0e161ed39df44102864325788647261a6e70ea4b1ee0abb57b57499bc82158d82336dd53f1ef4464c6a08126e138b2cc0892f765f6af85"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ab1fc9ee845eeb205ec13725daf1fb1f5d50629b14ea9a2235a9350a88"))
                    (acl2::hexstring=>ubyte8s "72d188a9df5f3b00057bca22c92c0f8228422d974302d22d4b322e7a6c8fc3b2b50ec74c6842781f29f7075c3d4bd065878648846c39bb3e4e2692c0f053f7ed"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "594ed82acfc03c0e359cc560b8e4b85f6ee77ee59a70023c2b3d5b3285b2"))
                    (acl2::hexstring=>ubyte8s "5ef322cb4014ecbb713a13659612a222225984d31c187debc4459ba7901f03dac775400acfe3510b306b79894fb0e8437b412150c9193ee5a2164306ebb78301"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f2c66efbf2a76c5b041860ea576103cd8c6b25e50eca9ff6a2fa88083fe9ac"))
                    (acl2::hexstring=>ubyte8s "7978f93ef7ed02c4a24abecba124d14dd214e1492ff1e168304c0eab89637da0f7a569c43dc4562bdb9404a018b6314fe0eebaccfb25ba76506aa7e9dcd956a7"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "8ccb08d2a1a282aa8cc99902ecaf0f67a9f21cffe28005cb27fcf129e963f99d"))
                    (acl2::hexstring=>ubyte8s "4551def2f9127386eea8d4dae1ea8d8e49b2add0509f27ccbce7d9e950ac7db01d5bca579c271b9f2d806730d88f58252fd0c2587851c3ac8a0e72b4e1dc0da6"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9f8c49320af9370cd3db20e9b50d3eaa59a6232d7a86fb7d472f124508d7968b05"))
                    (acl2::hexstring=>ubyte8s "81b002f15c4d48be8517f7ed89df302fb1435c9435efefed58f3eb8ea11910623f1eb9028a66e02121a7f08a7c604226f2324f483e91548dbbd2c441ab704ce5"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "4ab9aa069475e54b25e5688a52dd4acd134169c858105f01a0a1b134c72d4af51f8e"))
                    (acl2::hexstring=>ubyte8s "48ba5a63aba7e7bd8e420475331125a947928c67fdb00f65c4080d9a0b99c0672424e76a1ba6bd76dfe492c730f6f9adccaee7bb11571aadb31f6bb628cfa933"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f0c1d3407de92ef7421e42df5c9ab31d2ec0a750a9522869cbe4cabd66908d5823ec04"))
                    (acl2::hexstring=>ubyte8s "9e75c5bca2c2af1d7739787f46e1d981c4f98e493d0724b5252c2fbae3c526719f1d27e6ccd0d705240281e8fbf3db75b9b3205c1413436d3b5d140004b8cca1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ae8c9f8fb41b519b6d943833fe1c32d1c4292fb1ddf1dbe2eb227d9e14d31ed74ebaef12"))
                    (acl2::hexstring=>ubyte8s "042f9fd0a4ed3d9fec3655ae11011c6f2bc7e457e8812b6d8be2cd45fc6c432a94558c88f22c01439618865e8e49e509c448b342ca914b120344aaf7bcbdca18"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "da39fb86237f00303844e61fc6cfe779e42af53349839590bcd2f0e4cbbc279ec0b7e885d1"))
                    (acl2::hexstring=>ubyte8s "ecb43de8c233a731b38e30c5696f8876761b7ea72efe283fd07bedf20029f47c6d2a4427823e100fb087abaf22d7eff42a951c97c3dd05f48a20163fa4367cba"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3e7271d2070ef095394620c4b016576c150f34bea60784613a0f660d7fa5ae56872b88c58398"))
                    (acl2::hexstring=>ubyte8s "8154d0da634ab2266061acc123acb407650ffe9164a22de3fe29bf05393b2aece92cf4db00ea5b4341c31ddb7de151683c8a71b5a44d5c3175790feac67d18ee"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "311fb67f6a0784bb01a2d5a3f3092c407a9d3322319dff9a79f894291c5fac37319fb408402e18"))
                    (acl2::hexstring=>ubyte8s "1870fe913abb0a4b4f53b6581ae18322cd05328514556607f3f4d7b6a2ac8e9185d94d947d8b9c88e0efa66d89b59f7439c75fdadd1816f7412306ab2b59d664"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "7651ab491b8fa86f969d42977d09df5f8bee3e5899180b52c968b0db057a6f02a886ad617a84915a"))
                    (acl2::hexstring=>ubyte8s "f35e50e2e02b8781345f8ceb2198f068ba103476f715cfb487a452882c9f0de0c720b2a088a39d06a8a6b64ce4d6470dfeadc4f65ae06672c057e29f14c4daf9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "dbe5db685ed7cb848c094524c1723519d49dc66ef9fe6d57e6862a6435750bfa0a70f104f5d396e61a"))
                    (acl2::hexstring=>ubyte8s "2fa6e5b2c443a68050f093e7fb713bd6b18f6274c061ed61d79bf0688a61dba1940bcc30998276860943ab038902896d0fbf59b88b07c80de927037097150c40"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9fa83e96b2a6df23fb372895015678e0b2c9cd18a8542c3eaa2c435a76ae4dc9bd5136d970daff933acf"))
                    (acl2::hexstring=>ubyte8s "3a2c0ec88a3e5347cf0ea9c078838300ef7356f9a6c342063277c106b880a00ed2be205c13064097bd372fde38007bc306561eb4e74bba2bb20bd354aa690ca6"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "8a5a45e398bac1d9b896b5a2b4e3566b91d80ad20c977ea7450ff2efb521d82f65019ee762e0c85c6cc841"))
                    (acl2::hexstring=>ubyte8s "3c704620f4066d79c1ff67752980f39ef3d9c1023fa5a213a5265376b14a15166ffe069b51df7710d8907fef9406bf375d502ce086ac82aff17229aaa7a5a334"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "49cfffdaf4d031e33b1d28a447450545f6c4293b38d5afbcb9883976c014f080576ec691ac1bff70b742efab"))
                    (acl2::hexstring=>ubyte8s "8bcc4f1ea2b7862ef1591bfa73916665de8faf65439ddf5cc1be43cebfd5f60f205e835a2b186b675b041258c5cff42669316ce25b46a2f4d4218e102f0f5d6f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "2ff845d85efbc4fa5637e9448d950496f19d8d57da99b7bd3df7474822f0a790586736416714e364c6e1fae04e"))
                    (acl2::hexstring=>ubyte8s "236f6f4ed6e858c02d51787e60c578f731f694f8e52b5df4ecd5b04dff14c78e56bad1028d6f626c29d85aeee151a2a2846d3eed5cfafa9854a69fea8af6d04a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "cfca05fd893c0f005f5ff796f4da19ba27a1e729956b8b715e67ce4b2d2a382a72ec7814f2f507b1825209a20fcc"))
                    (acl2::hexstring=>ubyte8s "d80969284a4565add4dad6ab9b3bdf53446142f84aaf92d4b23dd22ee7241e6c81489ac8b246edcb6df9bd7b23d91a0c517f546feba4ed5790a2be6e165c1709"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "cfc425759a9c36bb9f4b32eed7767af6566f68ded0adeae25c7a70ca78ec09774d16c8bc357f6d6f7bd441bf62d942"))
                    (acl2::hexstring=>ubyte8s "b587a785cdf455cc9c544e756c1e306300aa3c59f8725012e68ab4d54020b6d227a164d9f83c905e86f8cebeef708a69f976d6e7b18b9bf78e9b98cc4a5cd1b6"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "097c9db919515242c99d973acb1dc4ed482768f974eb83b465f9f6c82503372006e4490835e2ec8f92301130bfb790b2"))
                    (acl2::hexstring=>ubyte8s "ff5a376f938e73014caef7fe3962944a7230d020b7087869ebe7ec70302721cd06fcdc981c893a425d05e2f99fe198e4db50a088aee2bf1263212110efed422c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "77e73d387e7bc80419ebf5482b61d5255caf819fb59251ff6a384e75f601ea026d83ef950ed0b67518fb99dee0d8aaef1f"))
                    (acl2::hexstring=>ubyte8s "c4c89cd882ec945cc888fb9a0127d35e585ecc14a75e4b5b3d8330538d22da28cf6af1ebec96dc247f109cd2aaab9756e6946a3d80db8363a4da3e6ddbb510a1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "317e5d9ac73ed0633fa18ebebbca7909ec3a5ef790478f9c38cacec44f196d895835b425774483043341381e7af2d383e51a"))
                    (acl2::hexstring=>ubyte8s "b10bb04491b9c0c334709b407cda1d503efb6b63ee944f2d366b6855e6e63e5b80115be4be7ff63edecdfb5923792e68123976d79212b3884dec2179d1fcf382"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "209461f20666a346fedf4a530f41a6fa280c43665767be923bc1d80bbcb8c9f8f93ad75782ea2689c8c5d211d2053b993145a0"))
                    (acl2::hexstring=>ubyte8s "67b7a328d9444056a52ca2f695c5d3f3baafb625a14fb32eee8ff26a40ccb296bec1771a826b55f7ddb6170d4caf7795b612448e66a0f19356fe505927149b47"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "5d61aa45c446f3bf93604b0511313b4e2f306d6b046fbd94797b926746836f2e1dbdc56124060c6ca9c911b1122192d112420827"))
                    (acl2::hexstring=>ubyte8s "d3931bde2bde8271ed18ca0b9148b12f6f16161e637e376fc961f65bc33bcacf2f6addf26a3eaa81b196653cc37e8a739ec5b3df870d8c38c8f28691c22a39bb"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9288c795bb0b86c0419d9c5637dcc37b39bfa18d441e3fbfca75bc0306e5432e8e7b3a5627b5bc7fdc424a77520abdff566e7f2bb8"))
                    (acl2::hexstring=>ubyte8s "e363d0e95d8cd18c384016ebeed6d99c4fa2768e2bd58fca019c5108b9cde1cb46f3f884028a55ce282ec310a10037faa1b16b4a6a669957f0b00f350bbd63d0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "780427dc164b2f69b8c7d569266f461e2d30c88c4cd6057fb030a6cf636f24e3c0d0db742a7b6193fdaa15eec50dfb4fae6ec7653c91"))
                    (acl2::hexstring=>ubyte8s "2964b009fb1bf996de12e030b9d6e0608ae8b9dbf2acfb9beb76fc5361cc104ee85c2a46fb7b4cee90848312da302de49afe61c546477e2b25d223d5e3d33560"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ec2a92e47f692b53c1355475c71ceff0b0952a8b3541b2938270247d44e7c5cc04e17236b353da028674eab4047d89ec5dad868cfd91ce"))
                    (acl2::hexstring=>ubyte8s "c83aca6147bfcbbc72c377efa8d53654ba0830c5a6a89e1d2a19b713e68fb534640deb833ca512247166dd273b5897e57d526f88eef58f6ff97baee0b4ee5644"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c99e31ad4e23ac68e15e605d0b02437f8147c44f5445a55b68a10905276cce8676481c33e8cd3efe322bb13fe0107bb546ccbec7b8b38d10"))
                    (acl2::hexstring=>ubyte8s "52992d45a88221d972958e9f2854adaa9a21d2bf7051e1f1019ae78004da50c5b55c144a02afffe539d753949a2b056534f5b4c21f248a05baa52a6c38c7f5dd"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9aa3e8ad92777dfeb121a646ce2e918d1e12b30754bc09470d6da4af6cc9642b012f041ff046569d4fd8d0dccfe448e59feefc908d9ad5af6f"))
                    (acl2::hexstring=>ubyte8s "994d1cda4de40aff4713237cf9f78f7033af83369ac9c64e504091ea2f1caff6c5152d6a0c5608f82886c0093b3d7fbadd49dfd1f9e0f85accf23bc7dad48904"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "5842512c37312511a3d8ae41f5801df60cd682d58b4a997342b6e717e94006c214813e6c63e75591f957a7ec301779838bec8ae3ed7febad0805"))
                    (acl2::hexstring=>ubyte8s "9763c43331ad0eb279d704c5f6e97e02da8724115026827f889e9fcda21f60fd230894ab35abb719890f3afa51afd31bc6852183b9c51059910af460abd2474d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ca14e2ea2f37c78f78ef280f58707ec549a31a94361073e37701bfe503e4c01ee1f2e123e00e81a188f08fa050825709128a9b66bb8ae6ea47e41d"))
                    (acl2::hexstring=>ubyte8s "4600e022a02258739f67fdd367cc1e662631fb087918768352062b9b3c8de8dbca0e9ec751b91f284694fbddb8d325c0637bccb21dd2efa92e48dbab2e5e9c26"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "647629c779b24c1e76f41744aba17159487532a0156a7d8264db50d645e9595ff81e0c96a850f2aa56c844c613a4b892727a9bfc3d3e20386766f805"))
                    (acl2::hexstring=>ubyte8s "5bc842fc2d3b7eb31d2d3044df3ec32af114feaa7cfc27ebc8630f46ab6f0c543f59b812e776e5303861d17da3f1f16097641f3b808d4d5cb3e483946409746c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "1c5dc0d1dd2e4c717635ff3e9b67caf957aec0f8f63c1b1e221e800a4c14848f4ea06e644e5d3e1de592ef5a8007fa3f07171b24bd07578d68963e5cb1"))
                    (acl2::hexstring=>ubyte8s "cbf1ea86fa5b3dbf67be82fac41e84cccd0d296c757169b37837d273ccc015eecd102b9ce1cff68fdc7f05d22f2b774734f62ded54c8ee0bf57a5a82010d74f5"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "8a555e75477d065b3af7e615475f37c0a667f73a4c7af5e4a69f28a68d9f4434776a8f90eab7f1d137eb4b22643c0a0d6a16fcfaa1bd62f2783546a9695f"))
                    (acl2::hexstring=>ubyte8s "c088e4a3d7da2f6f99a8f3f717361108872b8ffef921b383c24b8061d4e7c27fc56f4f20dc8f952a14043c5650b5a9e777c49c41cfeb3f2de97ee2e16b2c3924"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ebb3e2ad7803508ba46e81e220b1cff33ea8381504110e9f8092ef085afef84db0d436931d085d0e1b06bd218cf571c79338da31a83b4cb1ec6c06d6b98768"))
                    (acl2::hexstring=>ubyte8s "f33428d8fc67aa2cc1adcb2822f37f29cbd72abff68190483e415824f0bcecd447cb4f05a9c47031b9c50e0411c552f31cd04c30cea2bc64bcf825a5f8a66028"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c1ca70ae1279ba0b918157558b4920d6b7fba8a06be515170f202fafd36fb7f79d69fad745dba6150568db1e2b728504113eeac34f527fc82f2200b462ecbf5d"))
                    (acl2::hexstring=>ubyte8s "046e46623912b3932b8d662ab42583423843206301b58bf20ab6d76fd47f1cbbcf421df536ecd7e56db5354e7e0f98822d2129c197f6f0f222b8ec5231f3967d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "d3ddddf805b1678a02e39200f6440047acbb062e4a2f046a3ca7f1dd6eb03a18be00cd1eb158706a64af5834c68cf7f105b415194605222c99a2cbf72c50cb14bf"))
                    (acl2::hexstring=>ubyte8s "bae7c5d590bf25a493d8f48b8b4638ccb10541c67996e47287b984322009d27d1348f3ef2999f5ee0d38e112cd5a807a57830cdc318a1181e6c4653cdb8cf122"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "8e8ef8aa336b3b98894c3126c71878910618838c00ac8590173c91749972ff3d42a61137029ad74501684f75e1b8d1d74336aa908c44082ae9eb162e901867f54905"))
                    (acl2::hexstring=>ubyte8s "41672931558a93762522b1d55389ecf1b8c0feb8b88f4587fbd417ca809055b0cb630d8bea133ab7f6cf1f21c6b35e2e25c0d19583258808e6c23e1a75336103"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "52761e1dac0eaea898e0b07cd24f4b2e6bb7bc200ea4b0528842f17b87154559a2ea94459a0e480ae0bdf9f757dd4a335aed0e510138b024a04ed1d591b4323234dbd5"))
                    (acl2::hexstring=>ubyte8s "b826fe80494e19c51b42f2582b2d080ba6b90512f35f2db67dd7fd5ee532eaa16498afba08b4996cbcfdf8d1a2df6b1da939e8265115a48aefa42f38205db436"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3804ebc43cbea80c2bd7e4fda5c5515500cd2d2b846a1378dbf218d5c377138606eb3cb8ac88f9076f6ff4436f90717427c9df1ba052acbbe4585e98b6e8e0bf800f1946"))
                    (acl2::hexstring=>ubyte8s "17dd6d87bc6773051e52047fd444996afa8124b0483fe121877f98553448772bd0e7751fc655e9cc2d29830211015d310f191474ca6adc0477a187c03b8fe252"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "2249d698c4d807a8e7b4de21c485738959a0d67e5d2ca6f77983dfccb5dbf47931261e1f1537f3cbca253afb6bf4fe5e7672e1dcc860b3d6c8d243afe2d9758b375e955692"))
                    (acl2::hexstring=>ubyte8s "6af44563fc468d51182f6c3be58d45932af1d985c6f283976c91a9ff421f383fe21dc7322f397ccead583e26b3e3fda067976a7f34665df25a2ced7b4b09cdec"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "32a9c17033658c54f22c7135ddfc879de94d79593ef2dc7d3041bfa872738389864eeda27801794ccc4ff1fcb5ef3fc48833801d6fe959e3627f8ea1536ad00fa9c7d7d9f043"))
                    (acl2::hexstring=>ubyte8s "6a47699dd3ada2f11bc4ea42072b06cc20857bf164497df1285400c250f5848b6f71957dbdc845f5daeab913036661f69387893fc2d61c25fa59b9d85b19f401"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3d65f69a590a5baaabcd274fe3ef9e88920ffc7adf05c16d7b0f4d18d72bac1e94c3b3d83b8f4c552eb80e9fde3911403f8b000579816f02e1716fd627946031d0af0793e7f3e1"))
                    (acl2::hexstring=>ubyte8s "ffb2d9450943c24b5933c24812459b75d3d9f380344c9bc06fa3e17ee448eca2f98ff79f7e2235ccd9f9a8176f68a2254bbc9b834d6ac8d2bfdbc1597c432c9f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "76ff8b20a18cf104f6cdb65e2ba8f66ecf844af7e85e8ef2da19e8848a16052ec405a644dafb5ca08ec48f97327ac52c0e56218402c72a9a6dc1cf344d58a716a78d7d7529680bae"))
                    (acl2::hexstring=>ubyte8s "f8858144c6d709dd0689a526a548a43f17494950ba2ac20544799e8ea27201d78bce5b921e29a7b4029278e68341ef2a0ca4ba3894566b3c8f8950e3e545a689"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ca88dddfc876a12f45f19562bc9ca250f43267ab251a7f345c3c022e20144e135604078762ef5c8a8f038cf1b1d6a91709b59dd068396a9e971ab628f74886e765384a23607c1a1e6e"))
                    (acl2::hexstring=>ubyte8s "4f3d9eeef349ca51a7e419af1686f42795abde58a85335ce68d496e81e4436a80a61dc143a4300008c23a3e71f4ba98743195a3694a8d02fee11bd314569abc0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "0a78b16b4026f7ec063db4e7b77c42a298e524e268093c5038853e217dcd65f66428650165fca06a1b4c9cf1537fb5d463630ff3bd71cf32c3538b1fdda3fed5c9f601203319b7e1869a"))
                    (acl2::hexstring=>ubyte8s "6095c3df5b9db7ce524d76123f77421ce888b86a477ae8c6db1d0be8d326d22c852915ab03c0c81a5b7ac71e2c14e74bda17a78d2b10585fa214f6546eb710a0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "20f10ef9a0e6128675340171cd248df30b586557620b615ca39a00db534315a9012dbdbfd6a994986eb829dbe6cdaf3a37d4f59ac27298742c8f777b6b12677f21eb289129579868705f27"))
                    (acl2::hexstring=>ubyte8s "b4ead3f860eabbd36c770d66c7356f8107acd1485c7c94178c2eaabd50266d7645d009972586ef83ed43ed92882137df5117b88f35231b894ec1741ae7501145"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "995c8f747ea418f7d63aba2260b34ac3c7dceebb78438ca4b1f982b7db9798ec1a7f32622264cb024c0d9e60e955a6e1d677c923518851990a459b767d0f13cd803460f61870db3391b44693"))
                    (acl2::hexstring=>ubyte8s "a00a601edeaca83041dc452d438a8de549594e25d843c2cf60a0e009fb92d87abe28a72690ab657c8d35b43cd02d22ec0755de229d1f922fa6ca18a6d6c2aaae"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "0feb23c7e4a19bcbd70bd300d76ec9045d696f8c9687f49ec4154400e231d2f0862495150cf250b6f12f172a7d130f8fa5d175bf2f25e280172ccdfb327951701165302728a619aa2f242631c9"))
                    (acl2::hexstring=>ubyte8s "eeb6dee30c119fb1e1eb5c15ff2b32d8b9c7464a4e4cc6815cd251a6bae29b49961dd5c2fa9c44a9b142ca062c7072cbf3db04299b767789040196bf0c06aa76"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "ac59a110623f1a64666f160ed32926676cb5be25dd9d962f441951b0efcb5d6a67ac1a4eae473e49c6257860728853ff415c5e8ec76a8a462ecfd343eeac22dad820722c597332fbfd94ebbd32c6"))
                    (acl2::hexstring=>ubyte8s "f65ea942ae0a47e73b02b1442e5b26083db79307f64dd34a039c476faf18d5c514bb77a2c412a6074a7afc326ea66c74e5705fe2abbabf274333325a15b61fd9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9e3e1077e1333a1fb1aa633ccf2f746588ad426489ea08dff5511438b5f4c0b110d1a4d47b540a12b21ea2aa070578ccfa5c22fe0b743ec0cc621c6b3a03b75f4d3eea5dce89e03269afcd9603d0db"))
                    (acl2::hexstring=>ubyte8s "4b5c5df80c344c12388c723856cd06965b2190af652480476747dc2195ea3716f87c1762359583a5f31522f83f7833bec30f1f47d14540417dd463f5d258cd4a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "e881e3284c79d8f5237e699e4fbca84090c664bb53229f58cb0842b0436710c9b329d98191b8f030e9c1df89b03858c1569c6ff49a7c07c4a23a8a434b0fde13be4f94cb44ee629d5b44d336090d3de6"))
                    (acl2::hexstring=>ubyte8s "147d8071c7871ef9256cff32aa63ea031404fa5ee4ec09c56afdd5da919b0cc84a9d35d142c417715203316011cc620cd6855bb117063a5e52867facc680d5f4"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "e58521098911503de84311387d375c25929e6e55076eb6934fd8f2b1bb7b9667fbd76d5ee204828769a341b1f716da5bdfece6c62a9f4d4f988267fce1f5615540dbe375324eef607c910d976b45a5ea5f"))
                    (acl2::hexstring=>ubyte8s "f97ba056fa41f43b8e1987072a09e828c71c5ff6ad4e37f9ab6b89e2a078933dd23052fa72c6615b613904259e9ff9b55ef7b923b89bc8752f6babddd256e117"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3796cf51b8726652a4204733b8fbb047cf00fb91a9837e22ec22b1a268f88e2c9f133e5f8527f1b184830e07c3458c83a8ca9f9d9c6998760e610668ba0f22e22b656a737e978b246a17840b7dc4091da85f"))
                    (acl2::hexstring=>ubyte8s "c8a466199acbcbc93f2ce042968508c046901631e3118a2d0bf39a9b42b4197a379b3a86cdeca9df2de1a3eb71b79ae9bf2d6575eadf1878029c4093133f54d3"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9af608d031ccf309d7273c607a8e5e36840d449b55db5b13f03aeb9af49fa7e7cf1383ee2ed9c5a8b7515f16fb1c7c84a681590bf90f56597b844db5ebee223d78109b72350772f7c72ea996603e1e84f2ba5f"))
                    (acl2::hexstring=>ubyte8s "f0ded9495b4f64cac585be8a737cfa14247a4a81cdf7f01ebcb134ace71f5a83df2cd72e7773fea1e82beae17e13857372792c8231e2ab9fbeb633e399d5f0ae"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "d0df1bdf1df6203241722fb9c9c1cf7405017497ae154538ccf9224ad752e6ce1d4ae948639aca70cfe86b2b06543cb9914ebd3085aa3e2963f6e9b93d0b03a31ae26fcb9ca974eee016c091a6fcac37b21cc1d7"))
                    (acl2::hexstring=>ubyte8s "c2da3ea3c8a3fd88a5bc5dea2bc076f861abedefae5a5fbd941ddfd1c41cc3312eb2dc826c2c0f65414fe72ebee447d2f9b1a6a56302660d1f86632ee80a175f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "8cbc9480553acef7bcdba9716ea8d66b4131780917de2b0b048045fcb32b5cac054808e1fce6e94ad851ecb47fe6cb802225d3551e08ea122093d0078dada564212eacf1d6394e0007cc62a1d595ab14ca08a284bc"))
                    (acl2::hexstring=>ubyte8s "63b39b88ceb848188b37316e04560e75a5340ab8d417932d231c997e892b41daa69d9fe3e9a14dd19ccfbbfa01488c208e7b946cfaf16ca2b1bf7c8d8da4e6b2"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "38f184448f3cf82a54cafc556aff336f23f9149e612134b3fc00c8a56455653d88640b12f69062b8432c4335ad8f7ab4ff66cb7eb54f332561a36f024d92c3e26276f4fd48619628cff88e4b8e85cf14ca4767ed990d"))
                    (acl2::hexstring=>ubyte8s "9a49265fc641c59f1a91872cdae490d3da73c0c60fd59648e1d17dba1a647a5b95629392bb4ff5163d1a3cb45427c1437a3b2e1d9f030c0a8bcc5ed22da9e2ed"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "70900618b1e9e9db62296fb6c6590c9f10b0a632765c489c887f1ab7c07791765a62e38465e1be281b1d396c6e080b7ee3e6fa56a30b9799d0e629be153ee76f81bc6a3295aa61489bfa87d53a8ad24248a6ede0dfcfe9"))
                    (acl2::hexstring=>ubyte8s "1c8c3357ff1f8d6ac4defb3af462a73e09159e3a20c6506edd8cd3052df941c81f68c5fbb893912619e28640977fe8eaae8e9d5d4e7d5f132552cefab4540bac"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "4e6ddae0d805afcd10a055bce584c848d050fb29fe8f1c64b18e1abfe46b65782e6ff536e89d8d40928b41ed7371365c8080a9647f7532ce6c6d4ac21cfb0c8020783851ec9a7dbc3948f8fca7adf8b2a78c04d898d31ff6"))
                    (acl2::hexstring=>ubyte8s "5c2f996c779b91b3c4639311f54fabbdde7e2212b53dbae4828c8399588fc00d3b2ae60918aaaf6bb48bc757e52b2bcea84f5d15bf4ec25d5519fb54f6f26e1b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "696825f6d6ea8173ec47d0959a401c4ddf69f8f08ddd678a4d2ff976e3a4372bb39f4159845cb63585e1d4108d32e12fa7c5c9d7ce3508a7f53aca2b4bd951adbcd8984ebb7536563f5884c90bc5023b3316f7e4dc6958f743"))
                    (acl2::hexstring=>ubyte8s "3ce940ca96b00011375daa95c65f66907d69b3eb3b8d779e6fc971afcc05e990bc4c541f434590f6b18b68c080d0f24475a3e764e9cb85343301314ee2fb661e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "79ecdfd47a29a74220a52819ce4589747f2b30b364d0852cce52f91e4f0f48e61c72fa76b60d3002cae89dfc5519d3430b95c098fa4678516b5e355109ea9b3745aa41d6f8206ee64ae720f8d44653b001057f2eba7f63cd42f9"))
                    (acl2::hexstring=>ubyte8s "ba3d0fe04470f4cf8f08c46d82ae3afd1caea8c13bebbe026b5c1777aa59860af2e3da7751844e0be24072af48bc8a6fd77678aaee04e08f63395f5c8a465763"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9263fe75e8f6c7d5d642e2ca6a6eea4f44e9a0f249513ed79c9409ffca5526ca4491aebb1382057cc7c36722b0b6c3b15123cde312214f25353abfe30bca170568a8e1ba5408917403a01834080ab607c56a10d0265082498fe0b6"))
                    (acl2::hexstring=>ubyte8s "7736d7a7fc1eb05857ce7d88abfffa87f58c670bfdfc0a8031f60f379e4b6ad94ac8f13ffe28c697809b5cfac7f13be01e7496a85237c4025539051fb2e32fb6"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "78c17bfe0e02eb526d1a44a1ac127be082181452b625394bd6dc093a2cb432e6ee59c2f8b5503aba30dae41e1a1c6702697c99b2c94e94af48b00caf53b2e0e4e1bbee81ee282c7b2b35f58cf421a07e828d57a6622626af25835399"))
                    (acl2::hexstring=>ubyte8s "b56b6e343166328523e0d1693e5174da643ae83cf69c85a7b3c3bee247b77b84702069d9e6b4cab03bf17fe612009bf4239683ca78ca7e876aca7d07603ba714"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "298bb304a920f960447d8fd38b061bf8fe4ac1f871d8a0feb4549feb72ca694a5a41b6867d94cd5af77d468ad2f315d127b6c41a862800f3985e573e037740298e2c5c6186a9fb83609be2d49f8b4c31f96a2e49b56dbf09571b38587f"))
                    (acl2::hexstring=>ubyte8s "34e3878627904ffbbbd85266cc973c34f931e3cab5d4c31f841c553dd69f84838206067df4f9f3b9102001be19267151e673f5c2d4c2f8438a6999a0a325487d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "a3cf714bf112647e727e8cfd46499acd35a640dd393ddd263cd85cf6225f59890a0686dad1c54eb8d809b81c08a98dba131bbdd6fce8ff59d95db824d8831ea480529da739227a6e0f62b603b38c35cdc2581f614a31879b8be54aeefaa0"))
                    (acl2::hexstring=>ubyte8s "6f230ae4903ddbef0ba384c2e3506eab318bfd1a46ea76099f65a3fd529c91bc2865b9fd943e346de64626b8529f9db1377bf2c5e0129c66b50c6a5cfb364b3a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "0a427ae55ef3a7e6044a08cf6128cbaaabfd776c4e9374708f2ece246fd73603d2f54ac3e01d16cfac2bdaf713920d66e8f0a3d54ee68cff64267d5528cdf2f295f474d10f81173e0143488ac53fc503c444ed23dec63a080ce90c2443dba8"))
                    (acl2::hexstring=>ubyte8s "f6bbe5d0cf13ddf41c1436748a5d1ccae2948547b452c2171c7c8e8b66c6ae4de3c0e8b2962bcb60d3de3608479f80e455c9024d9716c38f6f1206861ab1eaac"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "2cbbb87511f4948efec3a61b511ededb1dda8b6ecfc0210c11e43a77ee32dc2e374afae4268e3d30427804868232a966b56006d3214037076bf6a265b72135af0fb2ef7909fea2dea412f7717446b276ff153753662b4d4148c02347e3259169"))
                    (acl2::hexstring=>ubyte8s "76897b87a8a1cf835c434f6d391c9e5227351af9d3e20a3389c796b98b424281a59068d9c8d567ec2bebc435b0126b059e2d86394a9854d6611e1c922f385496"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "2b23324c9992f60a7fc010159a03cb9a2b290df4fa6a82359b9af602f0a403a5ef33ed5da5b2caf87b77e6a4b93b650348ce2a7dbc08f8da9203d710b587ba5947c65e899f4a759f8e2b049ae7850a8e3e2962f6ef93ea4c631de5d78e729ec5bc"))
                    (acl2::hexstring=>ubyte8s "3beea0b373ed09cf1c919c51d86d642c9125e0ee81698dc4cbadf02e9e6925efb562fd9b87301a6377ca192be79c4118deabc450b54639000c2e312945451fb5"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "4022f930c7033b00d986c65ff6bbbdf9ebd0e58c52844ff658df3893c3202dc533f873d4a7f5a5f944419fb5528c9b6788479a1e891306acae7995fc06db70a59baa95bef7da79f5e793f2db7f2a55825e4fdb4a34884af881ded1089fd5334502a2"))
                    (acl2::hexstring=>ubyte8s "0358775bbb733ccc49e78f544aeee512370d480d0e13c7e8d5c444c423e592146b45fdb91a1b694d35e36b60e4bc8397fca8bb9790e619339778b9cd1abe3fe9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "1cb77ba43ce77e236b9fc925f589b1c070780a84f99e8f50c1ff846ac92599cfe91612c8178325bee642a34f4dffdba2aa2ebcf7064339829b26f27993e1106c139c70d578cc05f0e1a777cceddb10a2c67fd9675e4a009df8037d6eeb38f5fba233df"))
                    (acl2::hexstring=>ubyte8s "6502f46551a3fab3a96428fb97801d7a4aa2f17fef6603238df84e17c74309ed3d9489c8b16a9384ee634a3f86d0b3ba9a4dbc9c51ec8bd4bf8d61de6d3d87d7"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "52167de2d6c502d99fa10c27b2ab6203bdebc2cafbbfdef15872a43dd610c2362f796ad9bcb5528d95870058fa454453f1e6065b315d410a3f2650e5d71e69d78d9767dfb4accc057fd2069266b0f180cb319e30ded7535bbe52d24be151de4bb598fc5c"))
                    (acl2::hexstring=>ubyte8s "25cb3ed3983a91b4cf37a65193916c5e3e211b63e943e2f7b50a85d349a463b941aad33eff16561bdfdc92fda06a4e1d94b162de48f06d3c626940b31020925f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "cede6697d422ddaa78e2d55ae080b8b9e9356c69bc558201a2d4b0b3190a812c27b34bbcee3a62b781378b1bf636b372bcbae1fa2f816a046a0a649a5c555c641fea4ccd841cc761f38f777972f8c91b0324e71c333ce787f04741439bf087ef5e895011c0"))
                    (acl2::hexstring=>ubyte8s "0be42a25d77ac6ad995c6be48e783380bad25a61732f87cefb0cce1a769cd69081f494a1a12d657664ef2b4d9c41f2ee83f6e9a84327d8756af9f985595e7d3b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "56d18d3e2e496440d0a5c9e1bcb464faf5bc70a8b562124f5fc9e9deb5fee6544b945e833b8b5d131b773ecb2cdd780cd4e1bb9e4f1e3cb0a1d64d19cf4b30e44e6c2d0cbcb4e284ce50db7a8a8062ddb63f981d9026c532bf8eeddf8af5a43848a32262178c"))
                    (acl2::hexstring=>ubyte8s "982dc61c91a93770582eee8025aa55da8e9edb966bf5cf70d4a6534c0d53a2789a8c4fb65b7fed478cda02ed1e0d198d85c5c735b2417c5fab5d34e969fc8e7e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "25a7320dfaec5af65da4d0f8688e29e8e95532ecc16679ea8aff0f407d898db6922855b0e8901aa9681aa3dca617cb440764cdc7293fbeaf7f585b593c2b0531738e0ade7c8626b9995f4a84d9fc9b593d6bbee01abc53c5be14bf6956fd2fd81000dafc7c7686"))
                    (acl2::hexstring=>ubyte8s "749c928c3d5510925bfe98659025b0ed7c01acd4d59a9bf1c54863a088091771dc9d407bdbf83b0f44b0902e10349ba79c84d0981d5e8c4f5c733a117fed0790"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "3d7177b28ffd916e7e0634895833ba0bd9e0653df2cc4202c811536a005aec853a505e75db55d3c7107579041099e382a1feac80dde65d72368e909ab85f56d88e68d7c3c80c38f85bf8c2b36959409cc34ba8e3ad94fe8ee1927612d672d92141a329c4dd8a88a9"))
                    (acl2::hexstring=>ubyte8s "14a331508cd7d94fcce56a66bf65f20870a281c8442f8dbd4c2371454a2b66f8d0994a0b67692e771efc6a5e0b887acae7d6f4ec7338e1aa89f2abc7034c4e4c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c033e4a512297caecdbead892b11a9f7007af9a74bcab89e0bd4ffdd542ca03ea12e17a06c42bd43fc5f3f757fce4f6f5831997abac3f95676e1ebdb11ca43e11aa31e5ebabe18ce8d1bbfd8b02f482e1ce581b532e307e6960eb97441506c2ed299e1282523f41527"))
                    (acl2::hexstring=>ubyte8s "95ac9b7d22aa458921874c4b4331e7d64761853217c3f83c601abcbccd7e2eaa6ca6ce9a22ebcfe5046d52f8a09097f043ab8bc59243fd770090bb432c3155e9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "69fff0f1a3dbfb36e32f025819fa99ea9a0edaef73145bf7fcd05d8bb0a646cb3b5d5256d524856acfd2e44d6b72e4ebf1ff23c0ff6c56f821e782d5a15f7052a1445b06668eeb4af700679ee7ae26496fbd4640c06aa149964dfd6011df835ac13b73c8ff21151e8440"))
                    (acl2::hexstring=>ubyte8s "45d4daa652558d1c12beb0f5662c712f325b4c802fc6eb9ee039c949d002bb786f1a732712be941f9c5c79b3e5c43064d63a38578e5a54ee526acb735b9ad45f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "b2c439c97ab7c63736b3796324d68eeb7a471ed142bd9622684167d61234fff82f93f907537a909bc2e75a4bcbc133cf57197662c1af746ae8b81e5b83de05d9b589851de25d3c99c004c1dfb12d93bf50d450af49c428716f5b90ef088e3b6a6b2c46d3ce67b379599018"))
                    (acl2::hexstring=>ubyte8s "c48ec83be5fa669e6ec8db90aca9676cfe2ec0d5e8e7a2431687bb953c0a300be3db4075cca3bac4dfa4d971baf0fa1aff46639db4b238856ff36d1dfcd520f1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c016f522f26b7470e922b9a287e6d45f6c28813b68c1457e36d9ba266708272f9cbc5411f8db9d8bd5a9449fb6eb0cde7d4d03e5df019f2814a90ceed377c59d7d92623899bcb0268033073559d4d8de488686cbe3d67796e6df6ad4276d0b52cc62c49ebb58d7c95287aa6c"))
                    (acl2::hexstring=>ubyte8s "7402f1a99b47e102b3b73140c6771b07ee6c33b3715e9c4027c441bee40511b735d95e508baea78da26fded9b7038e9a53defa58448aba40dc1e62d7ec592107"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "a766b2a7ef916721f4677b67dbc65ef9b4d1bda1ad4e53fc854b0236440822152a111939e5ab2ba207719472b63fd4f4a54f4bde44a205d334a2d72cfe05abf804f41841b86d36920be6b0b529331ac163a985556c84511ec986439f83e1d7311f57d848cfa02df9ea0cf6b99a"))
                    (acl2::hexstring=>ubyte8s "ddd60f93a3babc78299cf763e7919d45ac6f479700e1adb05ab137acdf89c1521ecb9dfeacd091e58ca57a1db964a9c3cd1fa39192cc1e9f734caa1c5fa62975"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "10f2be77a4055771a67007cd8630e3230e38288499cb160380290174d66da57455b6baaa9785c84c8a663de41ed3bd544055b9170cec43cb3eb120eceaba1fe36e3eaa3fa4f99b425cd2519f09bc0282bada52d14ce625b1ded3b24d86b1dad342d2b7be322b775b04fc6b86afb4"))
                    (acl2::hexstring=>ubyte8s "a872fa33d463b3343cec57c20c66979c33e1ad067bfc703454696aab5dd0003bc194318f4a8ebbc74503feb7211a472dadee991efe3e38f21a1310f8a76eac80"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "324533e685f1852e358eea8ea8b81c288b3f3beb1f2bc2b8d3fdbac318382e3d7120de30c9c237aa0a34831deb1e5e060a7969cd3a9742ec1e64b354f7eb290cba1c681c66cc7ea994fdf5614f604d1a2718aab581c1c94931b1387e4b7dc73635bf3a7301174075fa70a9227d85d3"))
                    (acl2::hexstring=>ubyte8s "3b26c5170729d0814153becb95f1b65cd42f9a6d0649d914e4f69d938b5e9dc041cd0f5c8da0b484d7c7bc7b1bdefb08fe8b1bfedc81109345bc9e9a399feedf"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "518985977ee21d2bf622a20567124fcbf11c72df805365835ab3c041f4a9cd8a0ad63c9dee1018aa21a9fa3720f47dc48006f1aa3dba544950f87e627f369bc2793ede21223274492cceb77be7eea50e5a509059929a16d33a9f54796cde5770c74bd3ecc25318503f1a41976407aff2"))
                    (acl2::hexstring=>ubyte8s "c00926a374cde55b8fbd77f50da1363da19744d3f464e07ce31794c5a61b6f9c85689fa1cfe136553527fd876be91673c2cac2dd157b2defea360851b6d92cf4"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "9159767275ba6f79cbb3d58c0108339d8c6a41138991ab7aa58b14793b545b04bda61dd255127b12cc501d5aaad476e09fa14aec21626e8d57b7d08c36cdb79eea314bdd77e65779a0b54eab08c48ceb976adf631f4246a33f7ef896887ea8b5dfa2087a225c8c180f8970696101fc283b"))
                    (acl2::hexstring=>ubyte8s "3cd3380a90868de17dee4bd4d7f90d7512696f0a92b2d089240d61a9d20cd3af094c78bf466c2d404dd2f662ec5f4a299be2adeadf627b98e50e1c072b769d62"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "fe2d8ae200e6657fdc7494af5a12b2ae940348f1f983f0ba98febbe99c80d115126d57dbf37296765ebb5990256696588b3851d54c8fbe7ade98a6faf7c20b5e4f730f54a7f912ca0ac31bbb53d17949ef69aa0de40c7bab12a871a9b90f68813ca87af4256422a268f4a1d8ec3aa1a947fd"))
                    (acl2::hexstring=>ubyte8s "8025a8608df0f6a01c34cdec012d4cb25852e1b100b68172fc4e86ac8b7126b64859cb9e767a7e59060989cedbd925afc475ca7369bd43f85ae590e224e036dd"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "dc28484ebfd293d62ac759d5754bdf502423e4d419fa79020805134b2ce3dff738c7556c91d810adbad8dd210f041296b73c2185d4646c97fc0a5b69ed49ac8c7ced0bd1cfd7e3c3cca47374d189247da6811a40b0ab097067ed4ad40ade2e4791e39204e398b3204971445822a1be0dd93af8"))
                    (acl2::hexstring=>ubyte8s "615115d2e8b62e345adaa4bdb95395a3b4fe27d71c4a111b86c1841463c5f03d6b20d164a39948ab08ae060720d05c10f6022e5c8caf2fa3bca2e04d9c539ded"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "5af8c0f26db4e99b47ec2e4a01a786e77899e46d464ac337f175027b61aef3149848af849d76ac39b9b0910fe6594817859e55974fa167518ed72d088dae6b414d744d477974fb719c626da792f981233de24b7579d8acca510a266d73c0ee8ee1424343eaf6ffcc59c86c1becce5894072c6c11"))
                    (acl2::hexstring=>ubyte8s "09da284d5b6556508be54c8ab6c97bbd472995c6bbd585917ecdb54ea9167208daaa070a7b2b7d8e93ce1315f0d1ef8d69667429c44dc5ee1499de57b229a398"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "49cd0ba0df5bb3f43f68464e3e83e9cbd5d5ee077ffa5591e30f939cb30c93f7d454fb3fbf8bb05327a89c08dc4baf1eef50237317a405775357f1e0d1f31d9f0f0d98124019d47bf18363b1ecfbfe155c10cbc83300e01bc9ce0347c596b35f411e6d8229ad2855e42022b0373ade98663c6d6e9c"))
                    (acl2::hexstring=>ubyte8s "30cbf0679a97c871574d2fc05d7aa760c6bc8a864b7d246c39b9e812f9b7ff7b4ef5197dd5b69493306688b8564de1ad47d75505c913ba6a78788f8caf5788bd"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "a8a37dfc083ad2f47fff468738bf8b728eb7f1907e427fa15cb4424bc685e55ed7b2825c9c60b839ccc2fe5fb33e36f570cb8661609e630bda05ee641d938428867d90e00744a4aad494c93c5f6d1327878078590cdce1e647c9820818f467641fcd508e2f2ebfd0ff3d4f272393478f3b9e6f806b43"))
                    (acl2::hexstring=>ubyte8s "8e1c91729be8eb40226f6c58a029380ef7edb9dc166a5c3cdbcefe90bd30d85cb7c4b248e66abf0a3a4c842281299bef6db88858d9e5ab5244f70b7969e1c072"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "36af17595494ef793c42f48410246df07d05936a918afe74cd005e537c586b2843701f5df8952242b74586f83339b48f4ba3a66bdeb457ecdf61784eac6765cd9b8c570dd628dbba6ae5836b9ac3dbcd795f9efdb8742a35bca232abf36eb3b6698b2933965802277ba953a6edcacaf330c1e4e8c7d45f"))
                    (acl2::hexstring=>ubyte8s "158bfc348a30b4fabbe355a7d44bdc2122a4c850444c03f289003ce01bfc1ebf3ecc0febb6a8ff523d25db7681b05bdce048d11943ab476c1967cf6556c4a120"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "42d66edc5f22e0c13c25504c5101a5d172d2db7209e461efa323c0bfaed27e5f808042ea9c3838ea31f9b76de465225ccfbd0c09ca0d9f07e9a43e3e46c7693e00a7e1d483900ddb0a629d5563456dbbf299ac91f92c3d3c17b05d180e6c87c6c93194c39d90273fcf4a482c56084f95e34c04311fa80438"))
                    (acl2::hexstring=>ubyte8s "061afb119a3c60876e04c10f12ad0f4b977593dc5a2d21096a57e7d3f7d4d44fdef934b2c17d7530674e4f4a1c176dbdcc54811a22e1b8712e4192fc2d4bf8e8"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f91bb2e1a9c4cd96bf250426b3a6afd9b87ac51e93254d2dae3b16ec686ba80fb0bd7a84d218660e9007593075bc4f4c66567f0c7a5fd2010c999a8a0efa81f89ff5bfefe0fb910f0442e6d4a7c55bbb618c69a79a2ddd82a0938927f6fe3a80f04beaeb7c7636e3435d12dcf1c6bb6ed0a4edb69c9657fa93"))
                    (acl2::hexstring=>ubyte8s "6e692c8c694ee0a3565f37a299e0006b85ab4a821b20e76798220229f656efc6a20211a4e7e4ed77facde0d70e4d5d95bc8ed1d7a56d8df1446d562f044b344c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "d1eb961ca6a8f67c49b61e4d3ceaa2a1de6f0ea927b132bf987abdaa725b0e1e274e46830e99a2f75af608964df0dff9a99024fc6839bac5acd10202f921ac71a27fcda681aa3109ebf5f21ee3a849098ea3a551e844fae4b48b5c5bb97ccc802bc5520d68a14cb7e5fc056b67d889d876efb82d0e9a9a2499f1"))
                    (acl2::hexstring=>ubyte8s "39b2c76ec207120de4b320c7fe069e602c9c38f257596da7369395e87eb64b3acff988c1839ac269d5012c093f9edd4b7cabf13bdea7d42e969ab108269c6ab0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "adf2263200f376886ba7b6f5e4411d5f07f7d9d101590c73ace114bafbcb0fdc99269e87cd2cead2a1cfe5744394d333aba408a07e21f30233b65b907472e9e3c7d6e7aa6d2c47a08a1be7bb877913a6b5604c723384478911c339e3b5fe527c7e288705a89c95d970b443347897e79f6c522bafe62b11ef8f3135"))
                    (acl2::hexstring=>ubyte8s "3c23d2d8cf4db6ac6a42e27208180f37668bef5ee0a3f879483c8e604e7f42583f202037b8d242c04a87345b8be6dc8b121d6484b9edad0d73c894c1288f5cae"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "18e75b47d898ac629c48e80dbfb75dae1e1700b771165eccdb18d628bfc4063dd6c3839a7ec4cd1255c4821b078cd174647b320bb685541d517c579f6b8e3cdd2e109a610c7a921653b204ad018d0340d9938735b60262662016767e1d8824a64954086229c0e3b5bd9ad88c54c1dc5aa4e768ff1a9470ee6f6e998f"))
                    (acl2::hexstring=>ubyte8s "01c756b7c20b5f95fd2b079ab6a50f28b946fb16266b07c6060945dc4fe9e0d279c5b1505b9ec7d8f8f3c9ebf0c5ee9365aec08cf278d65b64daeccc19d3cbf4"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c2963342cfaa88ccd102a258e6d629f6b0d367dd55116502ca4451ea523623bc4175819a0648df3168e8ea8f10ed27354807d76e02ee1fdf1c9c655ee2b9fd08d557058dabdf8dcf964bfcacc996ae173971e26ea038d407c824260d06c2848a04a488c4c456dbcde2939e561ab908c4097b508638d6cda556465c9cc5"))
                    (acl2::hexstring=>ubyte8s "a4d2f59393a5fea612c3c745f4bb9f41aaf3a3ce1679aa8afc1a62baa4ed452819418c8ae1a1e658757976692390fc43d4decf7d855cd8b498b6dc60cae05a90"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "85360c3d4257d9878e2f5c16d3cd7d0747df3d231e1a8f63fddc69b3b1101af72153de4c8154b090c9815f2466e0e4f02f3af3a89a7fd04e306664f93e5490d4ce7fc169d553c520ae15dd02c7c613c39b4acd00e0c9a3c501566e52cecea11f7303dd1da61abf3f2532fd396047b1887255f4b256c0afcf58f3ae48c947"))
                    (acl2::hexstring=>ubyte8s "e8352ddcac59e377ea0f9c32bbb43dfd1b6c829fad1954240c41b7c45b0b09db11064b64e2442a96f6530aac2c4abf3beb1eae77f2bce4efe88fee1a70cf5423"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "c13e6ca3abb893aa5f82c4a8ef754460628af6b75af02168f45b72f8f09e45ed127c203bc7bb80ff0c7bd96f8cc6d8110868eb2cfc01037d8058992a6cf2effcbfe498c842e53a2e68a793867968ba18efc4a78b21cdf6a11e5de821dcabab14921ddb33625d48a13baffad6fe8272dbdf4433bd0f7b813c981269c388f001"))
                    (acl2::hexstring=>ubyte8s "6e56f77f6883d0bd4face8b8d557f144661989f66d51b1fe4b8fc7124d66d9d20218616fea1bcf86c08d63bf8f2f21845a3e519083b937e70aa7c358310b5a7c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "fd2203e467574e834ab07c9097ae164532f24be1eb5d88f1af7748ceff0d2c67a21f4e4097f9d3bb4e9fbf97186e0db6db0100230a52b453d421f8ab9c9a6043aa3295ea20d2f06a2f37470d8a99075f1b8a8336f6228cf08b5942fc1fb4299c7d2480e8e82bce175540bdfad7752bc95b577f229515394f3ae5cec870a4b2f8"))
                    (acl2::hexstring=>ubyte8s "a21b1077d52b27ac545af63b32746c6e3c51cb0cb9f281eb9f3580a6d4996d5c9917d2a6e484627a9d5a06fa1b25327a9d710e027387fc3e07d7c4d14c6086cc"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "4f05600950664d5190a2ebc29c9edb89c20079a4d3e6bc3b27d75e34e2fa3d02768502bd69790078598d5fcf3d6779bfed1284bbe5ad72fb456015181d9587d6e864c940564eaafb4f2fead4346ea09b6877d9340f6b82eb1515880872213da3ad88feba9f4f13817a71d6f90a1a17c43a15c038d988b5b29edffe2d6a062813cedbe852cde302b3e33b696846d2a8e36bd680efcc6cd3f9e9a4c1ae8cac10cc5244d131677140399176ed46700019a004a163806f7fa467fc4e17b4617bbd7641aaff7ff56396ba8c08a8be100b33a20b5daf134a2aefa5e1c3496770dcf6baa4f7bb"))
                    (acl2::hexstring=>ubyte8s "a9db490c708cc72548d78635aa7da79bb253f945d710e5cb677a474efc7c65a2aab45bc7ca1113c8ce0f3c32e1399de9c459535e8816521ab714b2a6cd200525"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "d5e378ae9fc2648f4a13bbec4b0935afb4f822f5fe0d5063053d2fbd547b33b4a32e7a009ee2afafe83d2ebd603568e4a38189b5d24d59e8953260f15f654ed4f42f9a39299d68c3eb78b09e83779d5718b433f1765d35350eac46493d194e84d1ce1f81c95b59725cab8ab73d369ab01e7967cf73a3acf1789227ee75fdfb6e40f353ff0484486542be053115db2896bab86c774f8985c4dbcc4c078f7b1c3a4c867cdc6580fe44a598673494cc0fb1f6598b1295768a584041fdbd14fa7b90fa6fe33f71b743b68e23f8e7407217aad9440cc8cad28152aedb82388be2de165496d051b292de6303460273a43508296b6237c07804335d2e81229f7c9a0e7761e38a3aaf7799f40fe9cb00457ea9d5b59953232676681fc71b261a6f8cd359293f5b21f0cf3a11b7f49cb5adb3c357bed2aa185d8fe8408192d6d3ed1ff465b590892efe03"))
                    (acl2::hexstring=>ubyte8s "a70c75b9b1f0ac2ed2c2797763ac9a6601d95f46889b00fc3ddae4d0ac6923750a108d79eb764e77ac07b7cb5c01cb4b3747dcf69ba3b35c51fb995da2632e70"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "4f7a5618870945b89f194e31b1aa802c5350326dc691df58708e34b48ce666b021d7c92330a69f1832412d8ac224156c9679dfedb383d9f9e13c21035d3d0002cfdf79b97ba0223cbbc833b0ad4cdd5229f2ddbbf6b650623d6cc9623da8a17d41db8e61cfbe772b23f4872adceb81e5f403535ff5f2ed996a67535994edf12a5f1230a494c946ed500e5280b5c8a82ddff369611afe58a85272e870cbd59a1012ce8509338a368b2c5dbb3ba2adfb33d30c494acca43896dbd8b030482841374055b81812c6f00c9e2bebe2096021feb69418a272aa356cefdfd22074ae91a8d2f1ef599a481c788dbe0afd54aac39672d401ef76d9f83175d177c9b72e2f6ab1e7525533d761d8e3603f14ea538904ed142abb3ff929ed55f4c6b17a72c685c3820b93463a67338756b2b033231a4f119cbb8d35d270a97791e8622340fc02f2093f9b393ad79161eb8c5897e21f7fc4b3ddeec02b736cc3ef04641c6179e825c319f6769f59fa5966f5957e573f9df0a2b76548cedd3e2158433dcb9de63f44f9be2b63319477570e14ee504b23b07cb2737a358154277912cd779abbeb1036f459c26ab7310f43"))
                    (acl2::hexstring=>ubyte8s "713d5c26de17e1440a36aab93f7cd8111cd62fd8bea5099b2b6bf93e470e1eaeab8b925c646e9e67ce01b03b33d2b500b9400e59f0ecdfb00dd7ddcd230cc837"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "5e0e84419c02ddda289aa126bdb40a060464aa58b88bad2708ab5f1e3df9ee439cb470e28eb627c6fe4904af033b6b01bf3536ba8748fca643c993d6185fd34e455a9dfe4b461cf451c04bcfc689b87748d9870bc5f6b91be004af18961fe90821a7147e1cdb44cae9aa7e6d50c579d068f9a535bbbc6deda506b9cbfd62b8daf744dc4b499d26b18dada371e7183773e73d991eb1c39f845b74bcd9964fc72a91d8fd4b1ab34a120771c4c2d4aa78ca8d4c6ab0ee32d748caf9bd29a90f9e61b50c8068d74638531d9fe84a5fa2c73c22cf20d1bc329ea1b93bc6a37ec9c5e82c886c89c77d79de98df18f0cf29a9316d6dc46b61eb7af7f1e2de2f5ca6c525bef3c996338194193fd85b9c6e66a81137cf5d652684f6b23b970eb58dce248232f6a0766379116edc33b93682824b45489cf3a75326973a5d025d1d57686d665238f8139f8e794435973aa754a41a33687d8f1930f8e7f71fcad34f039c2539e462d9542fe8520214ade7d52cf59e445f1a37a305236d524b977834d894d662b11b4b215aae27123f1dc7a9529d5ee0f41af62d19ceb1b1a71355b22075074b810c57626a097ddb7e8ee20b0dc3e370f126d19fb5223c7dfde471b2216a415b1cdc04ffa520dec59ba941e4a68fb35a1ce5d2e12147951f13b2d41d25980bfa49345be1541cd1385c15eb1a652a58083d0000dba56444fdb12687d9722aed90d2bc62eebe24df9c8b7ae897"))
                    (acl2::hexstring=>ubyte8s "d4a3b38f18b048e61686c2159c666469ce9a9400632fd40ac1d6ed33162b921b95ac26238af9fa00a4314a6046e1733fd4fadc6033f3ac33d2b8c202afa5f02f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "f516561edc530bf7aa1a63307d62812fb560dd5da66297298ffb0c4fd060a85a5712fb675cbd0e2d4e60f73a0dafab7f59b6140f8bbe63248274a863e1489f617814cd65fffecc700a9fe47a3965dbffaf350dbac655bfb1018ab071437df19b3e63f79f2dddefc8c61d51796a4bb8b6a027f1e5c1ef51d6f5834c025dc7441092499f9e4f241c3d6a2467bd092aebac12d2e719e354c49d44a8a69d99ee41c95d0ceee8cb99fdd8defa4dd07cca1be9ee8018dd944bbe0ed741e3ca5eb98fcd85f50b070b053aa00ae72c3a2df3f1761d77abb93c9ee73096962e1115f1d624f44c1e83752f7b9ed7352c81826d145a86ffc9354313e90b69eeadfb245ec0d7905951e273e167d82b2a742fe5ff2e9d10629ada127369a73993880ff4c8f3ed4363caae6ce1a658f0102dcbbd73e448a5e6452f0f85120deccbe8d5edf7176ef3b4b16ec8f8e1260c2d7ede728fe0612cd5f3bda31389446cdff5d677b3d5638a922fabeeeec04869ca90ef6c1ccaf38c75cb0a36e379549f17d699fbc3d5d3ef466cec488beac8ed2e8b8612e14903af0ff6a031bf88466fbab4a7dff0bec60a188fb93d9708ad82a3962ccea8069b7cea9b07fa6d5e21c312b6868d023844c34fc09a35d9c1000f6fd9670737a9c2db50e3c5fbb5a4ed826505c06bc1ed5602352918981abec87bdb7dc22657d981b487f6649b6102b296fcc8dde3f48203af10efbacf22a60db02faac291d03edfefd8fb6c41371f79e61f28635c65a254b2293d0940b922d895b387f1b16a81e4758b1a635701a4495a154ddc3edaf34ff3c8398c54a7da3012af3dd6eeed4a982ce87dd34485382bda285ecff01c1dfe5b350bcb73b3751d0f6af3f9b12040"))
                    (acl2::hexstring=>ubyte8s "1288010a5092c66dc87771687ca6227139ce5dcb48a79dbcacb22bd365f661fb1229493c662881b86bf1eca2071095bde47c96b992ee16d3e3277662cdbf960c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "bc4b7d3a380be067ed7a2a02f6137cf2aaa6f5332aa0dd34368d237e45db0a97d4e11f26fe206e574dc8f581d07e59076d5cd87c79397f0a862b9d147c4b771aecb83ddda0df3dc477f9577f70d4b1ea04d7ac34b9411640f4b1a14b0eeb93d0abd9c0a847d0b57612153742fea0ec64161f57a4113e0a9f2726166d2343c780ba4b8ecf1b7c3f34c7257c13a367713d4b3dc314aabac2efc892e6c0e2dc79f4c016919a436a41c83e2f103a86b3aa3d4d38d6cbab3ca8f87efef88d743d13f26643af51e82225f81e3edaffd10bfe2bc1187a133d4d2381f271341b01eefb4354d9d23b312d518d0f57c2aaa9f8d5d152e6963288749b54cbf382b4b0789baf31e0e1a147122f4fbb84b082380198f5a6a91171b3397837b28cca9ea96259fda561c4cb51592505c7cc8cd0a6aead606569d518e72423f34f45f42215349ff75a626301788b9d79de383edc1b877dda382ec96f4bbe02504925e8881c77d463f5eea68be42ffe71a6ceafeebe15702bc701d5be088e7e1549348f18e5b25f0a210a75b0bccdb0b6c798c54eb7dfef1ab9e9e14c451c000cf67bcf1a996daab484c0b8ba29ed1b10ea9e69442f6d6070b6efd5622bf1eefbf90191d55f311c28c61757ef5218a746b7e784174e482cf5681a9948e294b1abf9cfbe9260841d3090183a9860d5bc22d57bfc103bf7bc7174721646918db2e49feb3c99c3f115928ac949fc7aadc26aed62bb771f0277ac85ff32a46065586d935ff532b1fd6c175b346d65c92721e7a6a4a79421f2c46d41cad6a978a30a107c33cfa552328b87ab618c1974cb4161a94aaaae6a4e0d7e50daeb9e01cf3293026db439cd17d8aa24568f61a4218aa09d5cab0e219eb9a55b3757cb227237bfce178b374b34bc486c1d5f8581b2656b2be091fd6f6b30ab0aea04780413815c88383ce7f312a0282bbee2cd3205388bfbe046c17c3bd95804fad8edcf3a7832d8d9ae91ee38ea130a3f0281e0a26cc9b463b1a26f991e73f26f"))
                    (acl2::hexstring=>ubyte8s "34b4cf4ed7281ecb07f381e9fc3805072238e21ea0d1d2c5ec0ad3b23ea700247b8e9535de9fac6783f955166c4ca9a90f4caa47056fa9f313be75daf27ec8d2"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "edba7d6312144e90ec9eaace7576045a46e553dcb8ee5a9849272de841c8f1ea9502e3760b929c539f9593aa259ee3ba31064ece3483826ce7fc9cd0ed9ceb99b426f6c19c88c0d25bb24a57a0a85a91f6d67b6ceba545b1724bde7d87e7c1f211cdbe2c0c1e4277e79dd9bf2bbdf1485effe19f277a2ad0b3bbfcd956d0fc5f76a044a0a57eaba90dfd47c8110531453126884d2e2db92f6ff5fdcba4802346a9468f03adcd83cf937a9d5c98a7a1f137f6795f4ae47a3771a1e93ab71fbe117368a73f6d36505342bde2620acccab43714bbbec0c7f6508066c910d9d455279cae953a2ddf9c9ce26e1ec2ad9b32f7fb2b5650db00b4e363eca4344fc771f0ecbf60de35f0ee123b355a64ca3d632049e153ffe2b43c56e3dc828e642199d42b441011cc92cf1d14b89d30a8d9e599eadeb9b21c0e1713c8de6b065f0a563b20e4aebd3becc62537a2ef0e91983ec8a6cb9051d233cc80da99688487a92f8d944841c9cb285b926a9958e6cba2f8174450e1c34d4ee307e0b680f02627ac324f18d11dc16b16cc7645e1a09f1e9aed8e14b3f2d285e50fa4dcbc3282ab559319fb67a168e0803b235091245534adb14ac00ec5ac0059ffecb42232e35eb316ba99093d07ac2048385f715a350eb8cd41fbfa2421e8c74181e6bdd993684d24f99bc2d9e5fa7e597f1d872a87d143a83b6e80f77b65b73c2b19920dd9f0ab269fa9975d4bf023d8be2fdd7d92a8d7aa00c1a726009eb258b1f5abb7fc1e5682746a92b181b6962bbe407b399fb9fb3b579b789af9bc3ffddd240fc0451bd5a24087babdcf755fbc47f66748f62c37c309a4f939ade6e793754bc60efc0fb0eccba7176a7d5507f121f9ab90df2a13aadf8534ec664531824215d98b692270cadd73b29ed27d153413f24ffba2db18589ee6319c6a01a5330c274f1dd37a9ca33d939242171611ece68959708666cd36116ef72f0341cb0c9b2c30ae8442bcdeb4aebf3f954784806265d3d6d4f3baaaffc68e8f762d2c76b6ca8425c2a059e50a6c427b9139f55a4f2fc417d5ac681d2f96fea20eef4fda27a9546ad6bcfa1f077722e05613ece1758e351aceb75ef86a854f5212acbc9eb472a76508bf27f4727467552bfef17d9c2b2dc70a3c24a1783b8a85c9"))
                    (acl2::hexstring=>ubyte8s "ffcb3b313e576ea5b0a87449749b82c1523947ad019c633401a185fbcb4fd475dfe42289c5413f440d083f73b65abcf2386e0eff39acfafa398a3d6b778fde23"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "89eb3056770a6157f06921bc153834447c4b6d862d10d185f1c3f984cde5b81cc9eafe8bf532fc4fae3a89f41e14c52a0214fc1ab0cdcdfa583b8bdf8a42e3d3f857bf287e7e6b0c5f74f65dd22ddcbebd81e1eac022404875d105c0f6d1b8cea09c3bd417d6f5b58a7f319f396cc02834f8e69d65f77496d0eb31ce1a7b7e3248203a38ca22e472b8c3ff635bb9bb91efe2d1dba734af0df2f3d5b2fd267f2a794b84fc9e4a7e6d70a82b5141fd132177a86b4e8fc13ad5624fba14a7bb01dbba85e0c79c360ccb3a967b10f21e11ff53b07c9c24865d15a869a89a21234d7093361841caced5b79ef5b9497606fd540b64cc6d6f6b5ff64da767ef7244ab0f8cc6f21332fb2357252daba3d69b1f47282b226a76598a9fd873beb9a20862e12e9e37e5dca33cd356a61ac719c7be01316c483a43362438d13476a1ba56bc0d0024af5eceaf6954bac8cd5c477124a9ae910486deec76ebc151574ec115ffbb8bc121271208f202d7fc8e0b6981976827011ff64d61c8de2f342969bf2f741555dd56e8b0941f0438a8237c85de6066252461319a7f6239a344b499bd8b6340e8e11748ae3d388362365556a672c7ca2f6291e0cd2c8b25a5e9c087c1111467cc332aef16ea84a6df0cec41d12413bdce2379aaec528ccabe8b14dad6a9ceb0873d5786f0834c963594b0e710dc8e6689cfdf3801d8ad4f7592793b013bdb43a36668b542efd87aef3469ed6a827eeee8ce8c3113a636dc5367b7524caf8149a81523231618707b9256defee8ff2c7c7388483e2ba46f9bdf7096c6beed09ea1cb4527973e55cc1a861d8c09039e039b366da0750baa0b2a116de405fdebb4d89aabb5e623ee373b9bd1fdb6c19e24c2fce0c338ad0104921c156673868de9b765f277be90b4fb94bd17bf8848779de6f29a46be1ce0889e8dac7e45074bcfb6c3b5d781334e6bce807b4bfde6cee17fce06b565e9cd0997dc7f95eb0eb9b96f2a627fce5a2f1a417c691d784ae891e5d59299d0822b0a268f2b47e5eb3c49490084baf693bb7996177dbe97e37921c1afea6e7e849f0cc32c1c0ec64b67f1ccec9923aa6e882c24c805a9cd2ed5138bc36a9dd04f7d73e629ee44fcada13563551c445a4929847624ad6998aba0e9bca68fc276615ae2180f0e94f9609ca6b4c08a9776c1f42512b3d1787ae85106cccfe656c168cbd9c141f9895d22aea839b5708790579c8082ec1e40f4436a0d44b5fad541a424cd8a793b7c98525bcc5aba4abf277867293f33f07d956b406fbcef39c897c21a3fd"))
                    (acl2::hexstring=>ubyte8s "6b8be942ecd6e8509b88c0ebed80e0e2193eccf212430429e7d3d097285726e85e0d1d22d046cfbf37f80f922e36bb1c3b9e9d43cbbc477dc80eefc783adc20b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "014fd2fa6b05c4fca1a5c0753f15c940b5f976b41a40bf6bb14afe839d83a4676173940717bb7e746a2ac77f573e6744cf0002b78b5b7f664e22434e22d0ccd0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "5f1da3c03d633d35d6f28c1987ad25ae9e26b6afdebb2aed02f1719b706d20e5632b2005933db62129032fa3fd2304469928f6810af1742c27731a5beffb15bf"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "1c7665a61930433fcc7660b36049c016a7c646dd1e8412721314f969d3661554b97b18fd35945109cfbfc783bdc424b204ab19afaac855c284daf7f2cb333a65"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "97c433de3cd8f18b75e494bafc3c813e112b2261f1261ea6fab9dce0aaee5e5359f0e763f02154cac1589bb70c2cbe9cd22f976559032923c4e2beaf3f88f196"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "44a012fccd62e69ae9a2e9e46427fa491b88f66522cdc1f4ab5c5157196b15dba3905b99c460bf690711797ea556fcf4b2550539151f80eeb522fbc00450c723"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "b4641572cf72bed4513dfc2771dfeafc1aa6dbf7f6b740998c905f2f48ae471d00eb0e284291290360c9763500a584bab1a1603a8017accd15c3db7fac25b572"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "43331ce3faa34f02b379076a0fe38dca551dfd63148804b90afbe98e455dca1088742952f0f8a7f9d7ec0e38f92e95fed54c606d5799478fb380d84494199f7f76faa5b250503687772ee7d1bf24d8c2e314375e2741052f3ba91d3b63957233d56e98ed62d2861459e79a1fa9efdd7f451d891dcda7c8036a1dbf59faf9cb946d770aea6ea71e0946c02d08feb72c9ec4105e9291f517720885c688a06e320a05be0491bcf686cc0660aa7608a34e852895bea5ba507be4dc883baf9e1c30fd67ea8dca94e847c7617118500ddcca9e3953b23e92e27bc1afcf25b63c2a4cd559a4ae7d3da5e8611f98807e568a2850bff341455fd39f7249018c84c2c00ffaa4946e920054f101077fe9e72bd3f102811c74a26a4b6f5d72c77803780800d51061d5f1251dea11e3ce5b75d3e2ac1e4877523dfb7174ebf2c9f21aa627a7cc64b4eb83f4295cf8792ddf0f98dde10bdcace3fe596d358490ae01b193488c486a34b92eb4a0cf1e91427b9dc0ed193105622fcece683a91151faeb1889138b2e88a46dbafaac9ebf5b839089439daf88b413f93a08a0c784ac30000cd2a3270a75d9a732b6dd4c76c3521365ceaafc835f31a3718fa32a3f443ff87aa3c02ec6f56bedc7114771abd613a1060271d09cc1dc8a07ee13478de46b32e67ef318c071db900c9c53b4cb1e68c7ee719ae23196a52b81a669b61cf7525de4b6a46ba395993c57aa8a6c510691f693be7e97ee17fbf6533b2eec58e785c7d9fd6c9ddcbb25515cee742eda295fcb7a2ba5682ef07179113ecb50ce4f660f4ab5a425c86d206d70ba7972d2e0f2a5265ba7e980aa86223003d799d99973a2cb71f3aa1325eaef37a132beb7e82b120cc0ee826f90aa90343cdc0e27504f10c0b4d3913970155506350899058949a0b1edf1c9685d6f41924c3154240e58d7bef0bc51a7c6104d80e89368585edecfaa554a34e76b4494dea47fd25c1e11763aa81681eab26c372c2e4bb31cd36afd4a79604b8944bb287df8302f3c1a196ab85d8a987eaf99806c989a1fbf717eb836dccc73aa64436af499cab1508cd123bc83810be02828d7d69a1704dcd37a9ee3f1f753e7646ddad71bd80f15e0c93fbbbc82629bb68a9bdeb54ae8e153ff1fd64b7952bcafab0699f0c0af08b0622d3a5b17231dc53f4af97c9e52598d6c99826bbaf696b08b6cb1bb1d07a5664c6abd01a7d3e2fed773bb4299c7c1ea53ba7cb544c602117f8bc2cd55bf888ce886b822ccdb97adb394a8c9b5cfa6f544e129ca9f378b3cf7c221ba031d0e9eb6efb040fec0886c842c26548ac07528a0ac1e1e32c19ebda4ce73139dc6c416483439eea62951b5a9b8d8858140b90ef789de2297fab8fdd70f1b3c9e43258e0cc2b97e5b5c700394b913df21c6ef12c9f8ce6a475966883ee2aec77cf8cf608539c392a6f05ff0e5ab8cb3e4d993de98877377b95423e71d078236c27e1e2841f7ccade17b9de81d1e87c652ef23ccd39471bd76717d9521ec59fcf7399e5fb38b5932c364a6639b70b968f48e2b681bce2dbe0747e00ccdef8a9c93c1cc716a72ddc3a53205fdf140862e321d272ad1932bfbd43dd7d871af9898209d1507c73ceed30f6a3911af1c7288a432839f8561e0e975d9d162cd9b08721622247d5a47e0d37a83bc493553030ac364ed30a7d8d1295482794632405ebcfcf7676cb43278577ef50bff2482bc16f80aa088c956911e08ee2bed302e0fcb3d936a30456b09e17564a4bded2a024feaba6a026a799549fb6355e257b2903f607c89728e8019d3f629130fffc77a27f3fa8627fba0523fb498a3a0197c848bfe2951ae0fe02cd199d8d36fecd98811bc21a1740d695c61cce481c0d5fa51337f105f4d4a29b75ca8ea624834eeb09f9be63871003318429cdd6caf4daa8ac85bc81ecabdbb4dc7dbd45c43585598442baeb7a984ac1a64bcf9e0c5c10ba5ad3cb6fcb0db2354b58b3d0dcbd7cc8381d3f1ab880b22d74a95b07acee0ac16746a73859a42d060a83f7b135c23b216eb3d458086f445d366cf9797051cfa61908d7455e15570f6127114e97edc84d6e2b010d340d073934bb315c222d6752b941e3fda20b3539986c8a11cae62d3380ae4ea1020303aab6bf8c19882cdbac8e1854c7ec2cd16af679e961955472baec64447c3bf94096039f91166bf4329be2b6b00d0620328bc477b6a5c1e09dccf9746a06a796c7e76484c9bc922d7409fb438236b69060854b8d7a471631b850f10748a80d839f8321"))
                    (acl2::hexstring=>ubyte8s "a1a60fb89d420fc84c2caab24079d75ddca9a4676eb8677b3d64038fba8711203a459bc26259112c64138d4cd1ebba406720cf29cd1e4ac35e35be50464ef1df"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "3ed79e61d5843b36b6228023670b333208cf9bf556b1d6fc54e95808dfac2402ae06fb749e45883f21211ff41b28cced38d706390a398afa8f5eee760da041f2"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "838f976308b3e38471c3b4d8e5422312e4f56429b381f91b39bd27aa471e91a6e3ad8e313c64e6987792416c44a2a0e06afdae11a946af8b12c62f57af73cad9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "a163a7136bcc92589303cb15e26bc6ba7abd91b8ed194e2ade3d47f381191dc402e336c4c1aa8c0f9a2ef4149b1e3c2452d195d36aa3b96451e5059f5985ca19"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "0a6bd4a3be10c690ba358fd4abe6bb3b05c6e74762ddb16c96122156c6104102a0f4da5d119eae98c369b4a39bc75f86f76b829c7efa5d35b8c43dbce18a9c45"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "ca2928c7468e8e421b00fb128b1c114a983745f9b20521e8c8954a68178ccefc2e81d0f0c49255157e9b86cf09e59e187ad938ffd35664f9a3ddb471b85ee250"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "ba3ad050bcbd2febe8989cd0ade135303de9d9a419daaa647f0c89e2d8c78e1a701a4f9ea3ade33ec68b28d4a6e7ff15d637a1a803d2de1ff35b0d4298519cbc"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "a86c70c2063f12bddf997c91d1ebb57f5c55ca67614f397c1c52a87b0621fc4378c2eeb07732e5eb0fbb4b3c1fdde87e9ed71b50f83eeb02cba03bafedc84764"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "db7e2ece42b5186f84e6c1407593a9b9e2971aba7122b8167b21b31ea5f0bd21ff754e4fefba71c9327eeb19625702b64426acce7fd88467adc3d2b38a861abd"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "99820341517e3140475fc1a9e57a7f8b8bb85293ff7ccee237542e729531fe9daeeed96c3f41d87faaae3a7c6ecf308842b2e08aeb17e77a64484f41c60cd93d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "c1d54f9d273a339cd1750ffe36b10b1b9a60c9be1f2f36d238674e894d19b9929ef7be13043028a862c528e79026504c4a8b04a375a66d9d84efc1ee24325539"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "38469d81ccf58dfcbb44898bc73f80316fb511e466e19e4c4349a2c87ae744b28b872ce47970957a7e46a3dafd2ff7f2b83582c143ec9619b95443a3ad00e0236378e8fd40dd08f980cb7543278383b63a4a74592e0c8d2aa145701b4822a55b29e37ad60a830dff299ba7cd2d844b035dbb8fb5108aab0864e5d0393393bbe1438e8977bdabcd16e17de080f777786489527cdccfae173a6bfc0c04579b493d1b41bee6439d6fc5103bc8656ea7315e26d5e3d655c0fa481d52cb859a8477f817c6c12d82a11c951e9b0883785b9f7b9c0d6bac88087bbbe1aeb6c2e5946f99a1e4395464fffc0c736648e3fedaea4253c2bd6552cec31a21da74c5211556a82062ce61d0fe36b477632f8be96d393bb4bc81099a59f9ed449cae96344d57adfc09295eade7f315935c66851f5cb4ef25c25ea148f246bdb2bd0c3b088597319ca1f252e5025ab52d1795d98aef85e50f9efc3443e1d210d20535be830441bc6af2d0465f97941eaf5e638d116ed1a5db1604ca756ec4710e3bd2268f147bd4cdc36c4103eec8d87ff778ab9b20c3b8eff31e6bc5e14565cd298a7c13fb6d032ddd6e4bb94f1f6907aa7d3a124e80a5b5fcf1201b817c76789506c7f60241d0f3097d6c6901c15bad411a435d4033f493fb26f05f97922b7548ee0f22711cc3850d41adfb4dd536db82f6cd643e022ef3b664e4796bae0c2d0ae95d7e9028654cb16415577b8380f837bae00f9015da3cf754035245f9a208a29fb8f2a68d3db0bf8bad66d3f5c36c260d84467d9580d6849e55d06d9b746fdab92866fd73d41afd607c0e9c50869e1898c2f14eaaa92925e9e2dc2abb0bc878dfb5c2db01b0f0358b8a3820bf52f8d403a1f84c7bdc9da4ddb35fc681ed7c45b868ee895f5b9b2db4dfcda568aee124f52a43bc31a19ac823cf3f8716a4f409ed48d34fb65d84ae8284b3747322c1ce1664bae9f3808858633e66b6a2b8978c255d57742e3c127f9d80d4d50936019b0f57461aea02cf1d4409ba9a4cafcc9509b06342ee82ac6aaedb160f52730753880bcd963eaed7df6f37257d3d13fdee4148e0ff1fb6bc355d9d636c75603cbe288938a6160c08658cd08ea59d6b915d742eb4f4e861a9a951cfa2329a8608a398d870929aa5cb7942dddc5b9340f0ab4a68c969720adb8e3379bbaaeb6bd4cc80005ca2f40086e13964f9e7471982a06d0005a6e439baacaaa1951c8d9906ff860ae31e49a6a3661a70cc91e27eae0382fd52f26777aeca5f511ad0ab98080555fdef4dc41d88d08d063da18cf7df9e34689797234a1fb2f990310d6bee40a693e167788aef9fe73930345e6c0d7a372848bcdb3f3d752a2167c21df52518a9e9be32e7c6a34e73c7f59912ab47e89709a5a807e4d518fb7fdcb64672c5f6d8fe9d3ae29ec2ff3bde5e5403702e91867d884562399c70edbafaf3d10a5444f97b8566b13ad3190827714e79e448ddcb2cf87a366719e6c77bc66206e45c69d5077f9e9bc35e04a24a386dbe2f203530c0fd2fc995d4e053694aeb2c2964cd344b92cae961ec27d1eebb9a19783bff7bde9f4c65ec56c7572d8d778c00f5d8deeec6d3e43b04ec5dbb01bed1e723a47530e7ba509426049c6e97ec6f9296f29a99239fd17a2eba0d5b12bf2fa65f8de65edc5fd2b4ccfdead571341b07cf8331f4b217fb45f2faaa73f6877bda7721961086243574d11c080a2235d98628f36ca67a23a9c5d912fcaeb5fcfdaef3c74cedeef1bffdbcc8c0b4b2d9f0372aaa115883ac22226a6bcbb15808fc485ce9d29d1bafba771568d87b2a06aa435c11e8b1d698af832e6ddf27e792c2d03aeeeec915446c0a6566190991a64dd2468ebb5a7ad19810ab8000b66881956f133e6919e59a1edc32294b03efa0220b718a50fa773d25f36b0df506e5bfbe4c71410a3492981718071936144cdc7117646e703ad6c69e51f67a87c065338491db5f9bc32425a1e4699ea1e2b142821ea7a0e3c29e534a55d2317e8cf1c53163da00e26da42b0268e220561525ba9c47c38a7139d7cb72248998a211905d0cecf58ca9db199183cef9071b56b12c08ba3f92e2488a185ee9a319baf84ae35c54b46bb6b48b3f6c3b37bbb94d5c800fa0a322fd96be2156caa61b90e8273f6ac39f05d1a1031c05d922a4f8a12e9ef72178a18bcc7bcea2ef1c06ea8a305a7e603fc2c44a871295d0442b9ffa2f6e086445775491c90789818c4191866c1adf531b0b56562208664825b6780268d98c92920b208eb1b454a87f364d83e468d7c5bb464e6ed9e7be684baf495ee0c5c9d2e8c7b76a3abd4c8b7c4ce5299e2dacdea8dd7539b695e658407cdbb99f1c084dcd12e1ce6d0ab6530beb2322a44fa7c569369360c0fb26418c87f6b9aec4912acc5465625084dd27b47bcd13feb810a386bb708aff01c54bb536610614a7ac926af60ccdee1108300064aca7b24e2f9f88d2d3607c900d9d45f3e649d9aba1ad315e9eb84c9a7e96760f8884cafb4be2f6e585893e8e4460c4254b2a06bab7a2a8f5e3ca740888809fe229524da5ae93585b4fae4de5b3bbb8335950a2bb389a5723a2ac6d416e16014656ba44369edb379cf777b27ba2d7ab87d864b5956abcc751a5053f2f788226a4d963b69564e21910458b743a59db2488900d483296317fe5b27ef3b9eee9966278f7abaa48cf891f03939b081ff0ac9e49f461790320074992bf8fc61776d7eafc1b2c5d91ca0c31f73630cf3e3eca31cac10b52fbb2e8e30d6f1e403c4e73a22c793f9689fd2caeea1d3a76fcec06a9dc1b4ba5c40fa91dea36a50ea59cf2f4d2b9ef3d768e70fea5f0c1e5a223e86f47aa2b5aa75ed25ef3be2e189a86da47b483413aa1a56089ed56ce04770af9e9d317b25aecc8ec0deeacc0466334b5029c7d3f75526b85b78df767bbc5cac080470ae88bdb747e94a4ab648047cf0a8f5052f0d7d10766f14b7c97f2d28ac59c6d2ad6d6bc8a105add6121ae8f7f625598a85c4665a28c9df18d6391e87391a60b208323d6342ac76076ffb8846b44bc3e288cab57647d2545d1d5804dadab04ea93f85b72985543f79306866bbc382c087987efef74c3ef10f1f4a7495f9b20d22c3b64d8de183668603f008f99c60dc04355f875fbe6651994c8ee26b1ea618b935628502ffc087c84b10f372b77c72c30ffb031206ced2a635a3e2acc1f2edc6f508e22b38821fba86499aa9ad2c215e68cfe6b0cdff10906b17d9fd8c2f18e69327d959144eaad736187310d6067420d04c96ae6c6d87f24d595cbfc5c11692803dff2f2b8c95a8eafc214aca71c7d4e70fb77d7aa0a5465cf6cf76174f63555cb9b81fd6d27c9a056319ba0314a8c51d77bda0712ef5073a6ec6a2f9208ba9793998adbea6dc3525fbb0e11865a38bd4131db9d86ef3bf362fc6e8d3eafc09762c155b3b701104cd14ab4262cb2493ab605c2c28e72dc900c8349a301530efc40fb2aae5d4d7835b4c5d608adbc87723fdfd69db3e0fcf0ab6b449a625c6d6f352f50f463b7c3b51113bcdf91dfba0ae3a760577ca163bb479ba79aea32a5984a4510ac51a9b876ef1eaed25e82a9adab45a817721dd99ae9fbbaca1f3d7e1e8625b2feb08a201d56cdf4099e0f1d019da1f6061bbc26349773c5538be5c0aaf2b77e1306c2c3554e08d5add8eddeaec4a4924c02e711335beb0f5db69332e394e4bb13deb969c4e0b79ae69584b45d2534c310ac04643c3d4be10f27ad9d727692b326bdd6da1213c38cdb5c3c67b6ca3ad37aa37de5fa447b8391669dc9fea0aaac722d83dde8f012f0"))
                    (acl2::hexstring=>ubyte8s "bc393bc550a3705fcd5828ea43b536f6b2de8426bb7d0857dbccde4cf0ca37bc8a3c27a93096b4ebf7861843da5e4c31a43fd7155ce8c986ecbb039739ceeb04"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "6b543633eaad9b4acbdd18958c2f76d255fc32439ebf5b7125e6826da4c2165e4fb7295d2804e627a072d2382ab10d43bdf0c0e5e7ed45f3543c7cef42761f1d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "126a5b0739f20f4e7a2ca8f18d41bbfcdc4324af38125682ca7def664caa904e79b8ebeec1a21dc242673f53f46837b62311d45523bb1bf23fd1dbe8f23288b45ad745b5e9b643305ee64a97f303950bd8d4e31fe12bcbdf7c0d5011e513b5ba9d1093e418f4289fb5f1f246f4dc7c03e4d57995da2a0f51687975678d15c0672aabfa04b526751731cf5ee75eb995a72d5cbc4076808f51ed5d7374530051c4e9f0a49c2a12d1a1b9cab802996388fe9cb1d7a71ef4f2e6f20071c09d9b0fd7c227b934de81e5433628251f0da2fc10191bf0533850e01960ab98d1d4afa3aa9e976054393a59078e32cfc7223d9ae68de330ca2e5a03f11da610316104b4456389c285a307d4f5aa30de8140fe0987de52be9666db007612390068aa62ebaefb45074e84aefe14726a7ad6cfabf1fbf6aec33e4a9a82b027ac90fc1991f0ef4c13563844f8806cce27c94d370dfc4bbb3f664b40612fa84546ea7715586a10403bd7ca4bad0205a7b1f66ee0ecd07e174fd044b2f6af655250336105810217be04b48c02952be8e34e27c13210159cf12428b10722abef3712964053d25f0eadb81206cf77a41c6c4785cf964074ae3007971ba7adcb4fddd4399e66142afcf79d1cc11b1d14a928e4371f724694bbbe404509fa5fb68c2bde4402a60e0aea83fbf7444f4d2498016c65392fe81fcb82175903680015ae0877e0f57b8aa8df391cce17d1d07cf0de2939a636efbae764d4fb9159578fbe5ee935fe4d4eabe30147c7c30b6ec3b9a661d9edee220f154cbe553c352dcdc2147a6aa58424bcfec3c30acf63eea2fb435d6d96be4bc65f221e8c8b644a76c51273a77288d2833967d9c0c3431ac0bf9c651e2a103087cdce3ccb074e7633cc401c16eec220a79bd5d1112812b8c60c99c013304b9bf966396038365b79ace186bcdca098d8d12df6f3248438915beb22b87630d652c43249d2fba2f41a880bcc9c241073c0a8f6b08622621d6d28631f936cb21b1d9139e7d93dd37f227053fa0514c7bca2d5feecd432472914d47c789a1f3f81c60884fcbf2650665a0bbff83dd44cd944ff4d93237dbdc44d083e1615b75e7013d01a6ecadda10f87025aae8c8d27346ecfec8bbc2b4b56e16cb2df4c5b9850ce920327438b4ee474fc29043c67e133eaece2bee7637a4799a7dfbc8634748abdf5480d7257df4be8a236e3948901407f003712ec4dd0ff3b62ce950d6a7a84a9a2f00a9e6addaeaa9c8831db1c0fc2f4b0f3711212d7e68f448553272432a75940030510ebd074839127d85a47942112a61fc2a2da17645ff60edc1d8a255b97788ae511033e053b291cbe593317c4a18ac3ede4cc17e2711f711a37c6da0929f28b94e7dab1c99a0d6ecd1f0720156903e5d920115fc68e68b4f4bcfdd917d85511f9da97b02868e525ebf6ec377b31ba8f709ed0777a7e79e151f65aabef6c4921fc3463c48a6c584a02240ae55ec357c899a532e86409f9ff1b2b02d7f7ad78c5853b6573576f0fe3f9f288b126e581993c7f5f38eef7911739a420e5996387047c39c648f739b2fbb8a88f243c60e00842727ef27f0fe83093a9da5e3c5521e7c974adc8d292e897278e46c406fb1d2091c7b89b7e1aa752118158c9474cec745535437d504259c12eecd61ced80e0e612c634bd55e175859b76c3ae241090b506c3afeeda10ae80b50966b15ab5d064adb61bdf6b29fbd219ce43953a69b6b2c5681aa3d6c58fdda42e1f0db36496a14be1948a28bde04073d2ff07f11f06727b6aac25ae29a3dee3a329b76153daa5d28dc845f464cdd891ca208bb1f927fd0e128cfbc804938e5ab0ab37cd4b43792c03de9837e56aa15bfecdaae2c0c25b68a8e3ad8e1e06f2f29dcd165e1379243f8bff0b59095ea3bab52febec3c7e956f74d8c4663fa51491b1b9db7b7525aef56c5327c29852e694396ed6383504b10dbd94c7ba2e18fb2035940e8e37ef2034f00173db5aed73d3d0b1f32c73fbba29cd3f3ef1b406af88718937fe227e81e65d32780a16d8515dc320cb31bd0cb5aaea00b173991839afadbddf33fe7bd9f2fc0dbcb896ba881c5ae5e5b0dddfac7a5dffa9a8a5f5a07f315b8d95938d304015bbc94ea03c21f6dc25c90f991ba6806e65aef89f45ecdf4489ca8ca85d8fb8f0632732fd0623d5930b00b8ba68aca4645a46cb51b225b6306f1d800792fd42d42e5cce75bf96a0c5bc11110870c5ce8905fb8981e254ee3b370ed976d583c60b44f289cf0f823fbb6384844ad8f9e6238f549af87d816eae0625ffbece282cdbcc71cad5a0f099bd26e2774a4b50d47b1fd9667e2cb21ed107eb5aa724f45c8dee5ffb4d17e798122db254ce6fd380d9de1356db97f1e6eb5fe752f880ef3e68d44683001f702304a17c829fec89d384f489d0d66545ae4639a2507b8922bbb6a3adcab65e9222d9aac969856b7bb3c688c33d383eb75e28bba2baf6e894b9bd861d38367c557eda6f1f714794af09e5c24a06c9cc4ddeda0d71fa32444d21c7a31d354a22c11d4f190902af0e2fbe2acba59151b41116a4589776f74656dc1a223529b8f8dfe880cf0c84838203b8ddcb5c4898a10befcdc49a980c96b6d1fe2ced7c49a5b6c7b4bab25f8c9731e146e9f8926539424f9f7804c73bdb75e7bde9ce96c5d24ea3350282fb92f6c320ad4e00d68224be20609d5c6a4abd9c089e805fdff426b06249ce04c15d46c05c5922fc7eb50b8748595a2beb02e6321be81928c09e4ae3ee3946c117b430317f938c75f1c796a6b1d2d8f8b12bfb3f07e4deb29f8ae5ef2d4d944066605c1d70a3b4f297806c2241123e4b248da7ccb6daf584e581a0dd09ccdd09172a718c17d1c3fbd8a73603e263b5acacef90df0f118627fb4b23b469004ca590d718875c65e1181c069b517b0b65a4b11b5e83fad3ba2147b0157c342f069b3b32f81427a635d2cb634415f35375ba9bd7babd5f66b8fc33781e3e54a236737f02ba3e60dde8b04d87d85734830e77d2ec4c972613a93f266ae701b0a102b6c30e050aef52213b0a3a001d25e35762be22bb183822b573884f18cd307392eb711f8282bc4d09f96b436471aebb7903c840af487bdb031fc0ef59aabc620f9af0fdd8adb016c874d66cab5ab7529be25610d230d886bccb381250a13cb4a14ca870b093b1f48534b47b7ee128f1e688d258c9d7376a07005780a8b998b26f870140b23da83169bf6e39840ce9e0827ca2e13f4681d00d44850007fbb8a4a2d6633485b27e139b9f6a6e66e5e36796463073da839c00f6222a5c166d85d261be3616d2fa19e3435c12e2cb74fcce4a6fe2d642987242e91fae5a7c19e583d5c9e2ac401e63e88b7ccd5c5cc01d5cc77e22d1208ceade0ec57ac3e107c101267ecfbf7825436c998902a6c1d12f5a35fb572d1c68eb1b8e9bf45eef3e02927b6565162117a0b5950b1b86a1af1810bc6ee8b2dd335f43912985f54cc68e7ffc070d7be4fd640ca8887fc5a463c2c682603f280cf52d9c3ef8954ea0a74f10d3eb2db469de76b9fa8a962b73f413c4b259c35750b73d6fbe8f8d3394e92c8c259a2f8a57c2be04b298c87fb66af55172dfdc07528077033976d36acf950c3d1d9a79c557b12435f83b7d8b14cd4fb7ea7cf2ffba58c3a2d636d1bbfab3d969e075752d10b49093f6f91d153626083c44b76d082f470992bd56eef0bd497a5999b289892bd2ea67d15d10e6c47c0a29a518b6ea5956311c175e91cfbf61d7556eda1fa34f141af9b4ecbec665d83cc63728336de18c23880cd72422a902d8a487749e4125800891c530907cc141865a226e4d586ff2c71ec3480eaf223565900d086e2c54219d772cb0ecd281bfd02f8e736c75aa8e05266539b73fd301665f281ff72c85fcbff212f43b8e4d5a028ba2bdf98a3b6ac6e719328799646ec35aff40a009601b574c9f85e679cc1b05b786d65a60c98ef0aa08bce6877c8464c9bbf3d38a8c854ecbfaea535ba709bffb6346f9bffd006db7595e3ccfc34c31f483c0f878b2428ab84cabecfc862d199e61933d6f7fca235635a1f13ae3cd13228030759b795bf55bbc51"))
                    (acl2::hexstring=>ubyte8s "dc0465690127cf6bb6a066a8f400e64582e2540fb2794332533d37873f08dfe7408b2cba3b539138486fb75aeaf081229c525963c0d86efbd4c744f75496cef7"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "950233b2ee76ce1943d2c6bac53bb945b5e9793a41270cfd8607028df5c09be1eb311ece3fc5f58be0c5e233f42ddfa1b63c61326244a57c40a7e4df6a7e7f18"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "9b73214f933db062db8f1b46c60c266962c8071bdedf569c0e870c54730a0f97e200d199260a2988a651524a2abf9ecd8dbc831762f0a784f830462539ac24a1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "096a0d12871ba1ec81261a8e8b5ab587aba950b91580286aab0759586e04630b8a9e6a05a2422768cdf77645d2d144a7f7a3b8480968a809e7fd3f832fbd2424"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "4cb6d5a4382d4ab1edbff2c1fd938c982fd4270d33c42bafdb00496d860643e554a9039dc1d8ad1a6004f55874aecd7a73830123dedb3b42aa6547d6f4179ef1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "4018097493d3386e4adafea20ca962c9925bd5dd9c61c1a29198b064fdc313b599e9615bec715f9d3a2db986e64cbac04cb253e8ec977ea45ca8f1c9f802d965"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "825b192f69fb73b2716fb35e69d9c09006a8a2422ac7e4e065143c58bd146e71aae56d5d6574c39162f034d41b7e9e96102e24840c2d8b59a93de4f716636cb618dfb0fc4f61f50ebb7ad6257f512fc5e4973a4aa0bdf8c45964d304651581f7d4322b33b02ccfd083f7e3791abb3bc289cd2cbc6fddf2c87b714fa82f9275dce4c4262b0f3c1a80c24ea1f77625893807b6420400f6400d194ba10bfe41100c527fe90bf4f99dc5816759c81af5e2cd673cc447db71ab3c3adc1154598c1930264abbfce7cb2248c72a52138e950e29b046f9b089b4fbfde70be6acc9da5aef660ce626711a6fb09c7fd6015b6bd5f5eabb2a649129f8f727c06a3ad59499f21cafb0b0194e4ea6084e2bba08e1ffd91d859f8e7b3769d302e3e32705209fe04f1a62c368cf84d2147f9a651d5d537c528ba054d0451838805d9a9ae1e7346208354386e506dcdbc7fa6b2b22499a8791b575de17bdcbcfaa14d7bcf2511f204ee89d2218591d6a2aa7e633a00c9ca1e0eb6555d8064b4087f1e2b4bfc72ac1291bb6be89310e26436e53a61f0e21734915b9e19c8860321fdc02abd0af1beea1b6d883c1f779b81c38f9ab25b7f1a5d31bbaa9e9abd14d7bbe80c9ae72bb4ebfa14999471124c7bdeb216b3c6fd3de86e5add64ad7683aa27da435e079d28a8ac55127a3e9d7fc2c71130deb9e91c66b5cdd8f4474eba19c03b618ee2b28490b083a555ee55cff73b017babbacf106896691fd61ea23297236fe9961ef843448534a4dff6f7a779a92759a3201bc46e9ec104828a1d87b2b00c4e1c02fb0f24b8fcb340eeef80ca7fe74d4da05b92d91d7a3972102d23fcf484f179352722c1ca2033513ca9f67a4db5d913b21b2648b0d9212d573d81c67f6e86463c0938ac1763b75976a3730981378f8dbbb6d93df7825e3cb5db4bafaf1d7f0b6e0dbb344f1ba1f2238880e8c8d0d32443db46193ffa42f8b292cc2902364e2171c1150acfdca593a9b59aa87316466fe0d4d76b18956219e3d09dc3173d7630bdbff080281b14b69fb73ac91e7c5bd8f95007b400084de8b0312afe6c9779a1c866b844d64b81f36c19fbbb297775f813b9718494b7691dae111e4b60d8db0d1831e6c959027fa2953a37ff329c3735c69d8f4d35274601441cc22aa2761ccb23be067e3ec4606cb1a47a25b85f121a6868db8e6fdc673c08a0d12c9003b174816781d3e605c6ba37e55c9f3944993e9fa7c0b8a34b3f366249a5c1b7e63b91a7db037aed4b3513ae8939ec23c12eeac2c6e96c8df84088f773ba7a88a20d9b1d64e5d3b3091df16becbc9d066965d78862a01a7e682d9c2ebba9b790dbc9f938a1b36ee5c3f3dc169c3f225fb1b06d46f7af33031f28f87e8dee195ed7d344a02df6bce410594c35244c0cae1ea11484f24146aa81f5166d66e210414acc05ea562a3e3c52a0ca2e015dbc6024ba75c8ff10d5d345048e81d3302cf68906a55da6b8736d1f8cce87a00ec214485f09995a98611bbcc9c9425f52dfdd3735bb90bc4bfce4eb5fd1c8acb9be391e21386ca14268f26dc74f2567b4a963c9bbb119f33f2b422aac6c7773f8f15d0a838ac96f996de5ec8f88b198f00a38a51b1e6fe81c8ee371f5070a92471bfe920899162227e064ad852e6d4f7f72aa7dd65bf9076eb86735cc29f8842e8e7c3922bbf94558410d154a78cf17cd8367a870024bcd90083c0a845dd7c609d97484649be965b5303dd832203d6e64bcceb97aa9ded74b2ee054d01b0ed4837d65f830914cefc91e755aee8d5edfdf7f6e2548ab3880da6ce2a911da343c05fadf093934ba6ffe7c4201d24899c8985bb4708c9edf66d872b16bdc23be691bd22e4bc10eb842efca32818fca84d519c2402f98ed49f5cfb0eb31817023e83f8402147a87ad24f798b54e798b9671ee537fc6f02eeeba1aa7ee8ec457fb45db90c907c8282f72b05858cd1214452b808fe1bb44dc643063e839d7f733aaee9a79019a2c78a5b8a4998b81fbfd275e7ac81480f96e10d515c25ba1bcabc95ed6bd3a5029e1b6634ecccc11a99ea3a7379c76cc705c9b24bae6d2806728cd232acdd1ae530bc80f097298d07d814082319d86d25c4059bbcdc62cd889232cb8013b76291a5f51187f2b80531a3d848c031bc3571f81d2f6c8e0c92eb5237ea9e4868634f244e81c2a48011df23bf60126c7baaf3cc99d5226b745f75db229656f2f80766d550386756899a588050b9e16f1eb6f26602f7a81ba11d70557b366eb9703420e1829f3eafdc86e5e114bbf5e5ad40b8eee0fcc4479bb222e8cb34af47d7d4c3706c1c115880bfce6e91ce16742e2f8b8db94c3f976b08d4ba98f7e14135ce98a99718a6f87d2b5b51515b0bc4ab6db46a174fe0b21806cdf66de7cc52920ae7b58afcc638dcb4d4afdcc89d7e2be17aaf816d7141aa9357c2cf6c3720f770f8bccd2e39c0a44180b3b87f944ab7732781800eb8eb2a99b84648eeec536be812e5e918c737ddb9a8826fe6b8cf560b020a9b9a7f599fff39ac5fd5b194f75a7e5865203846900694478967d0b6109152725a798c3211ad22199d894f8ba2ae853ec9b05394c40f8c288c69a6a6eded60e14513a71f898abbb788728edf74cc6dbf9e2fe1ca81bcd84d02540048f608747ee5d09ac80d804baeaaf93538759c7e026b05aa8fcff5bfdf728fa74432c8d571c1192cd4a8761c268b840e40ae070230b899768cb7a28a0665846341aa16cde3615424ec5f557446b17ddf9c26f33f700e550d6840ab82e9629afbe764dba2fc4eeb4bb73c039c36a64a71f7e54a7eebf3a037cc73f8ecb1f13939e06afd9de30f4469de4669dc1ff2558cf7f4f207c8f2e2bb60567213859ac9e3d1c46606c12cc94608856f7b291b04f7445ad52230908462cbf4ff3e8ee6b2c26578fc25bd2ee0a9de02fec37babc17c871fd9174b57ecf4b65288b4f45c5d9e4290a941009cf35b07c4fcfa3e2af3e1684ce82d40ce8339c87af6bede85303f101353a4b1fe69252cd35369991e01ef8ee8e9ae387db1f8aa732fa474e7946c780ae494f9ddaacdf538e958b511556ea2fbc3e05aeddffb03451daee00cf2ef217c435a00bf1e1b09faa80fb98a56264b8e600ccfa573087cf9a724477cb08eb571583eed2f2dd9112176d5ce126cdf32a7a779be2dd7e8bd95b44dd4bd3812d9f20b87e8b23fcdb4b63df41c30ff60ff518071500e71636d242e72ac59f0423705447f1bd2bcc09bc0e4b2ca6c77725991d33aadcfab4034abfcdbfad842e877b5eb6cac160c7b6e475358014e3ddde871881dfcf8ec49bc3406d0e4c095ca4107e897c30697c3d454722de85fb76955f8886266129b44aac63499929b50b23b052348892bc56375ed297ba8264edd482eb95112a422faa96a6e7d2df0dd6610b685c7250cff21fa1d6088682b6a03681c04322836e8335f279cc5dd48a3ae6622f45bc032abc70be9ff9e51f774a15ec9514cd25055b67405a6bac92a0a54e652594d8223331b4f5bee038b11d280d13e89e8367bd14d910c74a11a1e7a1e374d9de59269fde66f7a15ff3d01ee686a81924e7c3ec4be5e45eec5f7ac5bd6011ba66b198c73ef124e5cdc8f502f21d19c56c6cd2296d282b7fd6d895d1542eec55e273e12fb00e1e0ace0aa85fc72ad96bc9851aeaf51a66a87e0f8b8102ba452ac612a85ffdf8437eb55b628bf24a778ab0b2f6a7803716775676b46b558cbca7760046ce655a25b0381dac2dfff66490d6890de57b08cb5b04a0a8d6a930c36a96df550182ae355daad3fdac4cb9311c9f79b93b51f9e5c1fd1fe1798a20f9d29d025b8fd26f12c5a200df8d650d5d9c7007910fb195065a9446f6ca84e7b07dfd1bb2cfd1171e86a246ba47d56d5a8efcb51dd8f664ad738ea8412005ff87cb3b8e03480b20e92f1ff816721a43a8629a04c83dbce33bab30c66930a9d30058dccbb59eed26e31d9fa78c82a03ad7b5dc692894519b31701f23d772dd0c8a955567522af3939cd095d7850003bd69073864a77370e74f50ce43044d0bf6b71f39e35182baf186f6970a8850b145ccc53289cd618761b5cbb2a67845d33bf634fec24c1cafcb21de17b31f1b356e838e5f4b12c50195354f60b0bbe3bf46c8a52886aa8f5e7b1e159c53b1c61c821da1231b10447bfc2e6a5d0447bd2ca1bebe2b92e52123abac3b997ed8d7b33b0bcc029c4e364301019b1c239428a7d7490e8c998498bbb78cae9396fcc6b5f3ce6d82210531133e6f60026139bcbd41e804dfed114c63390fed1f3ea6197146e5e4365c0dbb692ec837b47f90efd98c2c3f7a1013b2d69ad675a0d86d5691aee2ee73afa04b913c9ca0a6c142488910762d0f94f462627750e6ace6aae4738635eaee8e654837f25d396dc8c64d4879664bf50e4be1b3390faea7fbf779b3875f47fb9ae557c9daaa405e08857494216d1abece82b7356a733c368acb5cce0cdfa41f3778f0c37fe4741c03083508930b17e168213965862c2dbd15af242f91377dedf6c899c13c9c430c4ad724f268c0d1f16da00ec0168ae1c02f0b5654aef743dc5b2cb92fbb421bc3427acb0dfd1ec76341979dd1962ff6d4297879d9d07f846ed74dd58b46668cc57c5b7d97da7576bb86707d167f7beb5db77ef52b8060be19ed848540988a14776b01d34c8b855e6e9eb3030b4d0f2de13b775ec6492b5f6220693f0c1c2d217d910662787a9eecf11ce7f62c894a3cf3380a1d36953ff6dbf274693a3bb0523c80c181c84252a0e763779cf6becd97082f9fc6f8ff3f02ebe63e9a4791c41538bccbd7a85d2128b604fa8df65809da94a4b0d299e17912267e1058d098c8f8f6979dca95358ad2f193e93fb7df7f662d19d5c675d69e82f01ca21b51cb4f6bfac3bfe11b66004e0e82bc2897dfaed8361187292"))
                    (acl2::hexstring=>ubyte8s "c5ad53c45a03b0e1c8dc5a70031ed1ea16bd09be869d4505ea811ac6ea8fe988d264df3b698e08113cad1908b8710494e31f6b94275544eaa8dca047147f48ac"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "858deb41f50d96cec9a0850ed70b0d96207e6d5d8b3641fd09af5084d65a7c7b71fbb6fe04876ac728750a543ee31e94e388d42ca5aa2197fb006379dc67b423"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "52e4c7a5283c2c147f07e595e81832f5f31f29ec02b4fd68ea956d54a1f6c6bc953fc268df43ce47405362d3773f8f89fa024d92b54a1203f6d390bb88a73227"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "bd8fbb9bc746521cba8628670493cd851f43b6f0263f49d3b6259519fbbcd93cac1c7d215425a26a73a3c0f28aa18833a4811e3ecabeeac0917f2e18f7718c8a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "73dfee3b9a968877082b72fdec5a890239a55eab0ef441cf757f9eff51ed8dc392bda427abe60f7646bae3463610b0b4ada256f1afb419fec6f7f7b3f8ac3806f60c488a96e8c236838dcd6833eaab291af58fb31e4aaf314a159bc5c633cdcf0794816d43e4c73e286ad66ccd30a3e9e964b2f843fe2cf069a3333ca5c972a72d1df8d79a2146169b1aa659eb29bf7a80b867c90617b8c38b6766217cb84f15d06e54d083e968474665e637751e286649b8552d4fdf7a33182bdd7b32468ea158503e60e682545890123e0e52570535aa2e252c069c9bc778b53f52b1bc54d32d8b5e90ab3533d1e0eb55e76c1c7f6d0de07ece1397420814cd35afe062cc34b7e75b67ba4a2d3ba0b6d4c9aab5be98d998091c5e18987c523d6799a77f1b4b5f2cdeedfb09691d373c62bdf469e93d761c6e1dcf67c5ad9bcc7c18ee5c6ca08b218e2b57ba10ff0d45a8ca328229e54c3454d94fcbf02358411d349d42e5bca48f1428b4a449698a994ef84c46a517c3aa6359c48e4264ef65f1f69d77ae26133e17edfc103de416fffb4f2bfe865b434544a418f6e2faca00a165d443f0663ff64080154614f7194057d8b5f1f33934cc9fc2314cf86d4fdad4892bf0d3058f7f37ebe98ef52bfb240b9ad369153afe081bbcf9d7ae43e8ba336b8ac57e8a6da0a3365e3008072473bf9d6eac13e509c1619956e12a06fc696512da091a7d40232c675e737713fcf51aea6c0316c3bdbe196132b0943df2b013860105ce676fce7b88d0a167d7ec72c588b7b6465a83c9ea1d748d15713455e5d0e901c3cf646a38a09b0002dc5ab1687f350dca35c1a87cd404c0d529292082f77844203d86be0bb8a9d970a9af7baad8d050cbd9e024788eca91fbed39db930398180e393d949ad7e173d9c65498339a6ec670d049058653ad48af45cc4cbffd30c3b54cf1b290052b1864bcafd0accdf9b8e2a163134d2c982c1bba4a3dafec288e3cfd0ae1934a6f0e39122aebbd7a586e48d495167620708664d31c740bd868c1ccd5f0e94baf959e81502cb00da87330cbf149d5a8381e9eb519a8b97acad7a48c5b0c92623b861064ff1ce8455f32469381e6198c7b8abc341357d6a4c85f7fa517c4a47df728ac09a6645b0ca77df7c70cd4aacaf19c280949919132dde7993e9181e647e964ba99cd6bd10b893c8d90187a5009a23d295d43bfb4cc0e583b8052ac21651b23813bfc9912ea0c574e152f42d3f1975309588a4705196598ad93e1ab1d82954b4a18bc56e55039b6837fd893fa2bd7c70e21a5934dc2e990379ec6e8a2445dc55d57940a14e5164273f59cd58e5f6a8281e11c09536ea22821c98ac978537d7a02220d1d6552aee168a001715834596baabf7813e1c69949b23eb4b86658fd51819eadf8a13f067ca8a791cd1d53ab69d0e43f18bd72d5d93322cc1c36fbe33121f5ff01905328fc7c33d452a86468663c77fc80b0195ec1eca05a5daee339042b4f88a1f9371b472c6c5168c00e984937a134b282633dea25dde7e397b907b1e7d3d240a593e747007990782cf944fa078a7118fbfa793b2604fa15b82453209daa64475d0e95e2408319e8b5ce7460f4593a19e3831a9b363b1c5ddbcd273995fbc61ce7502b0233b1752223352e654837181d01a929f49faad422c65b8ae416ef81290b02b48e222c2b8c3ed57cf0494b928c1e11ad2da77baacd427785096aae1cd593cc356e551bc390cd5765ea41be30cf0266ae2e97d326c417c91e90d75f1f874555b88a14a7c5959a62f23976b77a4c754e35dfb7ddd1700df85f61a62b12a9eb4644caa7f8ba036b9f29c6315ff96c3f7148284ebe3239ecad50641f397ea24b46e21655352a4109b61479b9dd34972779f2f1a6a1d2887b8ff88289b2ebda2efe995668879bb93c4ebb3a585ab336f70b382205ac37c383475fa12ebddfb95b157172261597d2cb0f24f254feffaf75d224a3b407eb54cc7c8daa5483e4a79c347252d808a5f480a35987f6f09f6c6a73bd5cfbdb76a11ed78b86442b810cb703a5dec5874e8721af62e386591bd39d990b3521505e144100601b46de3f50752911ff37bb18f377de45ec4c60fc4ed8ea1717708d2d13fc9e1453a1c4a4db9e4fbe9b74cb8da14ad50c8c8f2ec944e10ee8e82ebb6a081959b0159f043a15fa1cb59bc5e035f7623fbfaa99ea0a1d81ae8692a4019e5a5edb3a4886c789675039fde87222975e86c2642eb0bd48408072fafb1a88507194c9bdd69f3418376a4d9e68c3b83b3f800605ff1dcf0917a6014b0dd77708b583ce3ea632746fee0e01a10500cba90016b4a9072847d809bb0481ae25f74f8ef290c7a087ae16f505fd0da670826a0b1174592d184e3a7e8622a5c84a30ab64aab75face50b96b217e8ea335c0605c638ed1c59370bb9ded004be428f49a79f74ec0fb296b3758f0b6b41930c7e029b55c8fa73cba7dc926151d4043c6bc8a716d7de9ae0cd3ef3ab2d19b0c813eaf12eacfb641d492b0001b2f0f699bd98e4581fd44c0c817646bdd77a71d8ed432f8d422812751a2f9178cf1800ee689ebf046cf9b161f9a7ef0a106cbe833398bf383288661b426fad8d4f570a8293629ee06856af295a58585a81f87f130e6e08f723234856e874bd0adbb2fc9e676deab6b9f22faacf12e875d1259ccea54f7294be02a16f34c427b51a33be8a0c460c4c07d51a2e7d5c0722a9fcfefd21c265d5aa2c57ae4fe95556b5e1388ea9756a6afb0856fb8fbe1d2bb1838be7a95049848fa9545b616badb753c453f266836eda3c92cd592bc0925690c42cd6667f866717827ebe91d0999f9de5f5fd6cf77f63737b65927aebcf6cefc7ca107fda8447e8bebf1f08a280d53a4b07f8e35904cc48cc08eda3c63a3475924bde1de6acebaa65fec5ee68ca22d3fe722bf33267de628c9db1ceda3c78cb2f9988682d641d068023f96aabde4e10071cdec2080f616ac30c2725ad3efe98a69a56873615a3a3161503a4f22621986def597b66641d07793d97cdc9a68f85fd3890a38928462b2fbe2bc5c509631438d2e344d1ced9e2b71748f1b6ddf33a3e597de3af03ce43d305b9f5acefdb2b71acc645d3b55fa3848484b7fa4cf25e71e766702f1003950bd2f45b304052861f6748a8f38175f1e96c91471f5a54999cc9937191b6adc9de0d2520d86590cd4aeab292ba9ae474edb5b8caad6ee095c9e74c0f5e5c9387559f946b2dc45da7fa1d4c2dae6973d5984841682af25ff7ff29d9721d6c7e76776e8965b6c681bc38e85da15954ecbcf20d7448204d9a6a477781c1564d363e4c634c36fbd3c3b50b332f1643c415d004ec999316e75694a8b98e2591678388dc6624058454ec3a7ce608b3f222b8bad5cef77095285e1d2ad746c557222dfc30605bfadaafc4f292e931a0f0d49b226d99d708247879aed5b9f2ca2fe6fb414f37373f844e13865524f206c54487aed53781834b3f6eefb248d95ba21bb60041d501f90a97a19dcd80920df7d84309148e3d0892e50687c86a45a1372926e00f200053f5f436e003e35bdc10fa99d9328853bf82d2091f1f087cc37678138ac0027e73cbcc99f7fe37939c98114fc7380c0ad1a26e3f5ec00bc7eae77045a55c62c18117879389c662837415852e7a2d01ac667a226fedb2596e3e137a83daec2712a65e8cec3e644e738d11bdfe9b19517fa593546373fddcb9e681fc97d1763bb9092a456cc0dfe1aa0e132387d105e3ccb7746ee199aa7af00bb96047310585fed40219dab43f057220a41e90c5f89fdac4a5d6b207c01d5ad4440c5ca29eed292c6f7000c58da111eb4b16e31efa6df3f3aff69e6447ac406aa96a9ece4b5b813bf8b3a499d09cd0969073468513355d6c19346c58480feaf470e0d45a13b74f2925488fd810e0f74afb9e82a24cdf61586bfae68dc92ea09b22d8c8f1ffe9db1e7e98892b5554ce2e15fd5f1cac5347df2eafd2a8d5f1aa8746b9403915da6d418c0b5a3aa8e09d6b65f9a49c3b7a5728e9baf95471404fdf64eb05da5f704dbad60ac9ac106cab2873fb1bc9023ad95c24852337a703d9cc04d6df7de594c3b2e4fb9f2996e0418ec8698a4c087c14a2687717f97e228e75afe295caae2f16513f47a45b4124a7c5ebacbacc562951233bf89f43ff85b703ec77f168c2278fbe6e57a0e7192125f4642d73f2f227d806287081bd30149b9d44fdb90029667622f9925b7826bd0343bc537c66e660f174b447860e1bb8846c3edcb639ebd213a4695f9cb471e188db7a859fcf3abae49569e676dec857b897627cb0bc1155ad6d45282d430176fde4262da2d5f41ff890ceb319d73dda804738456f30a3d68da41554d4cede62aa8549b24e211e76768e6b17379f842a24a449a0ba3ea73cfc72624b5afd118fd7e76a7c6b5bbfa7a6b6c97b97dea52decd51cf35a8e277140ffb2748777a1e3cc3211f3c12be099d0316f45023da6cd200339a718c72a5ca172903922e59648d08dc67f173788363c26e5df406391f107552925ba91b9e569f38101f5eef9a52d201288372abf6532beb4af19fa6d81eaf473d40896dbf4deac0f35c63bd1e129147c76e7aa8d0ef921631f55a7436411079f1bcc7b98714ac2c13b5e7326e60d918db1f05ffb19da767a95bb141a84c4b73664ccebf844f3601f7c853f009b21becba11af3106f1de5827b14e9fac84b2cbf16d18c045622acb260024768e8acc4c0ae2c0bd5f60a98023828cdec18ed8dc298a306c38d1ece01509f3265b5f8cbf441f0525097e8b48234bf69f65cf402c7540a023ed231ef95b222a900ea4bfaeec02c6d8b3b01648ad7a165237ca6b557b1ce287b0ea137f4ef54534070ee793695a9078ec89bcea389956878614ccbf917b61f8427b7cda870fdd92d2d297154262fc65f28ff1a54b2651afff12d6f36ee8c906107bbda399ce5e2cf0a430ad0dd86520841757126bad725bf1593c7959f16221894f5852ddad3172fef866b3321755491fd44fba009b42ec0b6c4fb9e901d7eb3b8acf70e94911f54c538bd0559c5740042b6df4a07c3e00bba0934d92a684b39592a576331e5a44672a227ccef3e595ffa1146ac1dcee0a70baa9acfd5c132b361b5ceb519984b0ee00cd2124aa8acb50c9e574fb19bd99c8fef5407faeedb28b796848bb372beb3f5bde55ed2cb140b60a53bba2df471f330208b09ffb8eda04315a06d693aa53d9bff8939ef6f3a68de6e1975f79f50b3d484665e4ee71124ed794be3a2baa7b5b918e62a095bc5d46e401a0979641fe465640e8d4d43eeba9d0cac76c7b86d22375123b988585e58f86566fd190d868eca08aa1e66932d6d3b14ecad3efd9f8cfcf2696ed42eadfa642324d941602cbaebb8639a00a17542afda32117051e4fbf243dfd255a559c49ac37c265827ba70b0bc618882336f43e1a6a729c57be478008cae6c74840bbe828c976ac628d7b6015bcb705612"))
                    (acl2::hexstring=>ubyte8s "c1644263ab1ff816fb1774369156e8e7001e1a164abc73f1bb4caede515eb2cf5708b21612f1ac475e00483f0552869603a7120c0d3e5a4d15380bb6d9aa9e82"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "27aa551dfa2fb1c0f4504f9bdfa68e50b4693d9327612a3fe8984c00d05d9cf6d3a0a8d0223ee5957504ea56a228e98acc6954fe35c7a577ba6be19abddc5dd0"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "e8e09fc57bd3b8e18b2726418836852cf1544eb2cf28c18f5b98436382d0b621953d2faa60d155dd28c2cb76c00ee93d3de56289b399c22a4fd82f00dd35ef27"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "6c66192d7e119cfb52e822217591996077095c73a7b957a0610ee815f9000fcdc7756c08cc6227798ec73b651666009378c879e47a2402e8861fd1cb80ec0369"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "2711cb980cb4fd62df85446940d9c5d0e382912c96e80ecfdcdaf7195f377130d01f1530b429e3658c371e7a91e3a738814ca8c7b1139d9847756f24e9d247fe"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "61ddef57b77c6c122d6128de0b45475d3af8acf74e521a74c46d99ca87111c33b9706083b11417b9401546d4398fece83053e47081dd0ce69587a79160fc009f2eee5173a00ce240f12e75d77ed3e2bd86ceff812e3cc24cc8b9aaba54d45ae847c088cf8c4c15d0c9147d8fe3f97205c4d0c4d15c5784cfe56d69ec6c8272398d4ccd3e7ebbf8904f3026a66feae4e873c0fc0913169a97750aadd0afee90b887fbea687c43d3a90be838c12b8c808d1a086fd05b614bcb00f83e0e2500fc8dbd35a5f691cbed4285a57a8b1dc88e1d410fe5cf3d20e1940c9ee44dd6a1fde1b763c890c23c8f8446d0b5a94f8755bf703d26aec03ba21923666735cbb7dddcfeeb19dd197c0197d5cfb1c6c9af1e9c30faf914ed9ad783ccb78e6f0b2be2a5193761869d748ba2ba0c3688912931db08ae28a9ca26c7f990f922a39ee6556590ad6732560c75cd7eed0c97b1ded6a681c1791662c5bca4f595ec788dcf7e43ecb7f58a4d2d0642a8dcfa9c31b55522219cb503ba60fcab85f1b197dfcda468b3bcd13689bd8726c040ca3f1795f57973a5a417dce0d037c8763ca89cca10fb7024c4dd12a540a44e8142276d68ffd3fc450dfbcf2b8070458c5452ccad0b25a8569627f1ce79e1db072bfd3b15bb5746e4b9579b2b547d849cc03d785edb0ee5bc0fbe2d1f7a08ee394c688784712ed8eb71d249c694870d0c4912a8c4c8ddaebdf747ce7f8a3f10816f115b9d697f3ac0fb14dfe687b56946b77962c91cbb8cc1156dddd8f3aec8465a9fbe48ea73f5cd1fc6687276301fb0d219b0541dcf063032d56111fb7f9ef1d6bbb96d3e329079a790e952a91fb7b4619b994e1542a473162a6212ebd71ae8ab0d52f4abe3932b59196c6e68abb9b92ea6269d4a3beef08912e9c30be1c594a13c93c57aaf864dff0af8aefbfcae27ffe35d2a05e36d105335fc0ca2c2265d80f184c8f3bdf2465f0a96faa9391e659dfa48dd8d61a83b3b3da6d5faa5c6ac0fd32ce65d365f7e42c24d3abc8b8fddd092ef9e2cde27da0bcfec0659b02be7178b93050241535d6fd925567a198bf65f535856288273538996d887080bdf7860c0c10e2807584bb0f826362c0c991814e78debca7413107e3771404fdf6ac5b21bbfea18aff5b0cbc759c2198b42abb5f396475add8f77b15dbf1975545238d544947f1a62b3f489edc5e38bbf0c170b22795020e470f2ecaecbfbc491213790e248b77548187fb213727fc8cf81c5de1940f063b23de8f5d29ccc82bf5e2cad9de4797388aeb90ee0697f19e4291bf683f30e6d6e7ec195723a76f805927ef21f6eac5a4bfd0b64fdb1ccc6268b821a0698928441bce49436aedf6e8c4495280df13a73b88cdb7e11524c17bc2f63144a3963f1537dbbbac5fff12b7bfcd91712720a06f8e39222c555e954cf9d24aaf8dd6bfccb4cadcfc8359279fb43b286d4fb045b328a9919c1c18ac4865e2fa5b88d91ecd2ae229ae659ac9e349f883efb9310b98f0a562bac9e17b00af9eacc3150bd0b8f2d82e4c87226e34d9bc3083043166a37e9331f2ff5776082f6bfe90c2ed79f2b5036751588e8906fbccbaa1b12feb451b297bff1cd941c17de15c2f02fc8f1cd9fa41576a4daedc18c08ec6308b0e005eaead19b2e552cf1375321865c76499ea3358c93cce4426c92e398b02d15357d1b931ed420a2d7378ce3cc76bccaefc6e351a1130beed69756c274ac8a13fddeccfc6e45ac636712622c8b6757ac2a6ce8611e9f66e0f26b610e4afd26937e53868c8995d98022a943962802360543b838cbc9dee7a2de103a30e04b097e0201d0ab366db0e25125b68969ee60bb82d35df96a79951ce2ad698efe3f3cbacf1332c13f75376f3bad446d18d3ddb82fe5df51cb177837466a77926e80c6312daf34366a5478102a6a73aafb5bb6e29215443db36199d95170eba3f7b7afc966fe5f7380b37bae6483cea8db2cf223f38c1624e9feffeb4ce74d28608c006cc1273ade4839462f3221dad07fee99629972dc6e3697b2567ff7268b72a6c00185524ee08c8fe37ddc6c2188b14ab487cdee1f8262d4bb0032c6fc33f5aef770fba921e2125c5a7eaa8b2a3937e74a79d72866ca355c21d3ad7f5a88b14bada16b513d4aa349b11ce4a77d4cda6f6322ff4939ad77d8ecb63748c5f26d26eb7b1c752a8107fef23dd1e7a327628707698d792093efef25901a49c7d9f3e7b8e8583b6ef0062225cbba74233497d93f36c8dfa4e4664ad34b9cace7a509351dd065cb93fe685d50ade77585da410d1271f9a44c09a491fece0696aebe78b821198c4e2112f4597dcba013528027cb890368efa168716bae01a031a28d0719ccac15ec9e9673bbb30d2d76119012baabd25de92a6a9580ecccb1a54d788000267148cc94f1ae2efb5bd5c54a71ca8280c720ba6f77d779bf17831a37e8a91438506d178a8b6a585c9de229858b17cc10caf889f8f800fe30ca5cb7e6464cfe0932cbf409cc72e1debb9ac94b46c14064c42f7e4e53d8b646cd85a4e4f82f66a1fdab8c981c8d72899d46ce230a8f85839b0bc3f8f2274a0908460ab9681bf2099b5507bfe382b3662c2d5f963f4a447539d57d7e3bb5618164e611ebfcd74d48f2118c06891a4b7c4e99a52b4dba244384d4d769984aa4b7c7afe0041e2d1d66ec2743f6761de3d3ecd0c76a9ca7cef46652f73f70adbbb21be7a08c35e30b6c70de551612532274ef1beedb94d9e990c695058b2ba74d7a46d77809a9c2be616580eaeec683b1663dc242079e171e208be62ad23000021832c79105e192df4db5532907f87737b1c00dea75202b15aa9f8d8d9ee9808493c5fe1207471b153947a03be812f55d5e5e440fad84db7539e9d25876bdee78f3e9c624984224092c03ae48197ac6cb7401aed83830f0e6d251fdb0e3141489418565ef3f3a77622ed1c777239ab228fb90946e9d8e32127fbf55bf6c57b85d8480249a8497dac9479931f08dd8801cfcf33ef86404689e471b23ad99eca39cc4c44b4eefc216e6804335dcdf9efa517d12f49f16fc5c35334dafa2f23c9def093822320921f360e0b71b63a253e66c89d5fe8b3addd47d493e41c500af6efe5cc4f16f591d38e06fde2bac5d5e0d182524a3e3580f991e93d4213e4aaedd70c450ff2c671faf0d369eade0982937fb5be72dd0f8fa73c119f89736589fb3ed4ac49592bc8d1e903ce0e826dc1b2bbd3206e279a306290ee293ead5b3ebb81ff508fbc9dfd04e108a8d25a7af9cb307b3690b07e09b85acf68b6342e2fbb4abcd0579f858510736268bac8a9c149581e4f8efa138214d445dca4aa66d366f3cdaa15747b07710aa1ac4c5f828eb9ad2cf9cd87032004a7d1f36e728dd03b07825c5dcdf6ea933136e1eb819dd8a8aa27c3b0c9b56a0440045b981f1b9cc4107b55a51e81a5136192883cc1442572d9bf1bed44b2c690374d73a612889f8e8929246fe893dd6e26552da4a12dfbb4b63380e78a83dc44e82dba0d0f6d6ef6ec1c5732beb5ea0ff9ff30b7a3a3d1faba2591140d91017eed25bd5148326189b3b55c720360517556f14fda4b5e3e916015ab174072dcbac0b5ec075f604dd36a94a2c2491baa3ab7bbc9a10aa5983542745d55b591d494ed5ed493d77f5a509b2ad89812406accb6ef3c9064228532dbe21653ef867f89cda0f58b23083d9daff85c57d18709a7beec172cda5d7c3696bf748650b933f60bfccd51e4635df7237daca619581d584a32f7daefaee39cfd8ce6818bb2a88602f2ca8c7e493209903b47f75c2bfc5ae3744d3f169f61f28adaeac9f30b33a65d038baf6302604160b1ccc00c8b4d2c04f095ade1c76dc3c91a0fc4191b6bc374af7f1f25275f4b3f619994ccfe865849684fbbc38f6d8d32daa404806bb789e46723605730956366db4ce9ffee2e4c0b1bf9fd5e8fe7bb529d760fab94bd5626caf7275bb68936817f454427d2ab93d4bc22c6f52172da92c2c45a0cfed3b4a63b88b8b511c14119de1117d75484ac30a938cd39ec85679deb3d581732a1220d4956b8f5ead40f9a683ee970ec4011a963ca549f59a5eebe973f9e7cb89c3aa664cf8c909a17ee7dbe3d3da6f27450998c561187f14adc8f2749a18ff92b4b1c2e71c807b8238a5e308a98cc1f97d59b834a68a63beabd7a4bf9e5d38cd4efc8f6dbdfedbd7d5f968fa56f62bc013d29d935ee5ca20c1e2a45beffe9ec3f95576e7c5895229bdac0fc2c460d6066e3cf0e97072dbc355af9d50734358fb1e821f5c1b6a08ff57868267d911e952f65f583fc479bf2ed10835a714748ffd234c31d4a7e924e10d4e69f17f538d2d13037e9c42429de99bb31467b8d86a52727b5558514dcf5746ce6b699d6cea1e2def0908d2149e7d1d78e39849b8128171d732ee01e030ea6bcc31d61ba527adaceca28c5c39ea71bd214a42ae5cae128a2771a19fe87021e2116340e0466d62f8c2f12ec8f7a84de574f06ecf0c678e2435b97613aeb6851465a8afd697e3c679399df8fe97009afc3b7e249410657ca6c7b377457a2b9d698fa82ce4e0b26098184ff2e6601bde6a3c9b6ae5afcf647aa606ab1e94f6f14a590cbef26801b8bb29ae23699c21a49640f04aaa33cdce079a5b1a844341091d897c6ee978f9574c3263936d913a71ea0e25cbcbe28943b56b5acd60d0316e4be89f43af58c2c4b1fc154adb898ccfcd70e1f53005284aece0aacb9b83f0a73d3e1e452d508387544f6d519cca4245a72cccd1c2e3d56c14c2f41c1ba4a63c5a53a07c4ba864299bf313af969c825ce10b8f6667c6b63095d9dfc9ca7b7ab509eecef1f2a78e31f0afae6cb349749ae1d5d2bd08f4f31ed8ed14d98f50959bb4d950ab08962927ba4ce691da18428a10523cdc8eb6b5be43c63a54516a1f76c2dacdb276bd5c9460936524f35011b1d64976cddf5424993f2bfa43fa123fcc9bbaee0f256450ff4c49d030e53a0c999c842898e516673635e6fcad71901d27a7405ac9fcd32df0e539d2ba9a1e58c9c8c9f9438644a71d8bbde79a9e2ce20e4d36b9e8e2511aef932a11ccf94420c9c4e191d8ac250bd7ab796412692eb60af862b575c1283f85bee3356deaf38a22f694995e3e02bcebfbe21a1ebaddc0a20d8ef476809ec539ce2de7a65574be5bc8a9b9de2423e1e1e5bfb2c57982295ef5801ef607fcd4942bd0cf7e2210e5fca1ea985f46d0a342cfdad3108e6518c61801dcfa3e8a931d3e9494fd3a0c278de81ddfe98c512a9523f92605e1ceb663ded8d68c0d8f2b5c6d41de2b83b3ab97070b3584deed9d6248e552af5b052bbcb0cfb3b35c55c4d9219e5711e64c9aa2657c433924ee2432a6660013e2519e59f4ac1b781322841dc374a5b64b1e1b346a38b10f469ae98ba543d83b5ffc37c31438b8e11a3f96afb98b70229b7257d77e06260f2c3a11f8ad46812ebb4c2081763b2ff959f420176dab615ebc90761209af4a40be15c7fe2635a4f4dc8a7e1b847d2bc5b8884bde459c1f7198c09f6d73fcc6aae575312778d9e896acfd7c1aa4c5524f20453e8bab255363164afa7124b2425587a077fa0bfaf61b12ef3f0540dc4c9e777122a60610a53d1d75b0a5859c654a8ddfc2ff4860758bf5a6f264bf8bc2baa7551eb7be23bc06978be992fc81d890e07a3abf95d20eee3f6bbbc089985cac96395b473b2741c66bd2ccbef228432f66b906c15b19694dd786c29f06cbc17b2e6400dde4e3db85819382b3d05a4c3009e44f8d03fd2dafa53f308c9eee9a2f552036022321334cc11532f9fcdf9e9ab0423cacf346031d91de2d0cb9c78ed1f5b3253d5fa7f12213a20516e07fdf75431cf6a595bc929e83aae0a9bf204667f566f416013ebc105f802394e202b8c9a9baa4005027833a63ec72ea18bf69fc3d81cc156fae3c1d8df8b3fb6c2fd09fcc646f468b4e7df919ed0b572c6ca095b54cd4c312bf1a518552ded1e5cfd0fe93627ffeb0fa29e48df2d17963f1933c450f35a75239feb42fd6cba0b5aa5fefe6fbfbb026f32e02a846efd72615e9f8d511164a98e38f4f73cca3ef43882679ad790be55501365c50308bfadb9f4bd24d9ba0324a8676293a7b8d47f931cca4d022d923a5ece6f86ef7ce68ec5860d095f5fae9d2ea5b4a6a2cb22f3e6cfa78383f8af57e7be8e0ebe4dfab"))
                    (acl2::hexstring=>ubyte8s "73e3a0b552605e08c1ca390e9b8d498fa2679a4f2f29bd80ca7e40a86e20a5ed4a9b8707db0aeb79e251c8c0f4259ecd25dbb9c82de838ae953e712d36991546"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "f054933142774856dac63b63f175544298a3d36874bc37650a2682201f8336b621cbf9d95c87a9838181c1b5c015dd0c08723beb82830326382b5db51ea9283c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "221ca8cb81dae78612b8a7c2ced624bcf7dcba005250e194818805acbc64a41ffee4403672405ce5e7e81751736ef1e7addcfad82509ac4f0f5ec6100eea39f8"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "9ecb44114c5cec8af0b3a10498d7cc4e0e60fcc3b5bed7d5f7df23ca7269ea002235f97062a6ddf3d8a42e37e48122a8f869007da8855da0bc62468b53b8fddf"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "bda8abf3e35a759b9444370ca91c0b845a93de37d4bf5b814bad94c6733fae356a3d90c779a4cc9decd5722a07373b8edbc42159562a5eb764962b9541ce784c"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "64fe3513e438e5e0a977586a1780d30795367d6bd622386b4fa8cc9a91a88bfdd5666a5a39ae2a9f7f2ec67943edfddb42e96e5ab81aa0cdca312f9d60f8c9dc"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "a3481b7a56b6ac361213e5a69c0e99594c332567a0cd5181992dac5ac6ff236766768e75968685cd44ae68ba7a03c1d1471bdae070deb899cbfc1690d4a69b47"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "08b3276cdb68adb64cfdfb6d54002354108ff18c3f901c1fedc48b5a44c81bbce19e2a231a319191fb5b9652bc43c20d80ef6d438f7227e72ed57c7fbd9c79d1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "e0c80702526891b3335d5a0b3180d47025a76469a296e823fa7d7228b9872897799dbc0a1252c7b7e89043eff1bf3092ffe7b67870327324cf95742fdf0e4122"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "8ecdb553cfbe2451feb4b6440056099dd6511faef2ac7ba92b646c061be2857c5bd09710aa56658d4e3f566850b680b23f2e01344902fe0c83e8d3c8714c8fac"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "49e91bdb49492471e1474a3934cb235d877451e4eb447fafe3a01f6948e06fe71575eeece2e41d9634110be5369199458c4f2f68ca380079845138ca5c4913e8"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "b1b2dab94b3db9eeb1d579e65f06f07004b98f214026e4d53b36917808cce2f338b6d195287cbf2430f3867bc1fd5c1a38a4cb3ec741521da718e2ca33c9359f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "b4e49d08be92b07880b5f9496ef0974a09f44d57ae3dd5ba0dde1b498f09eb219bbf3019dfb28e5a4d79d12c299462cefa246cf4a13fbdc81f719972065d3ac1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "19cf6c6dc28eb06cd9fc0a58c87c8af1aa605ae7fda192ec6ea3c12ec3b6dd529d22c45ee32a00597ef615af5a8e790d3902818d1bc3d33341e648b349c5377e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "cd200254e5a8cf6e0923e70fc2cf3ce26ccfe07366a1c5e14944959078dfb803cd0d0289132a45bcad2bc290605f33e90184d306f35f0522e3d1a315da594bbd"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "52ef6d43536b1fbf828b21b669fd459acd71e0ed8c67b97e9a7a79f145cb3a5210143d692ee36db3a09ddbd18bb26f8c50f473919e38e07614b8a0cff6e11012"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "398b4e5c2570d043e9a4a037b0a01be41ce56d58a26a6014ad7dc5bb435c86b4f8e1f8d9673b0e4a50e1e548352475af8b1081f3b618274a77182a49a1d9e877"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "f52e19999a5773b270b3ac58b152965e6c5b37ebb082d1f7a6d723ca9cca7e2faf4bbcd94ec326f16d350592ca7c3210675ed12a03b559ba66f9e681dff9ae71"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "00c0541af4657e4296c95c076cc0b87504e771aecc78e2259b3dee92b0ef30f59025653b8119436a86aba5e7b2155ba2d112532d3542f5a9684a6277131ab387"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "da9436bdc81dc6aa3be907377519049371e61a409f850375d3a503060112f0eacf8539c1d98cb3a302f9c125c78ab6e3a3e695575fed3eed2214a0b9e3fc6b51"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "3fb5d3d8d7154294799650b12df25c71f508c4b0bcd4da2ad6e57babc98439a1fe21d4ef57c7c0befedca85d517fb9c5d5a705ffd90b1a5b9f1717001c8ef547"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "1333d76d9642fff25a78e1df287d433bf30c61c2157870e19abec86524bc248c974a61a14546cd6edd382910b9f269a913b09f72f44534797662d3c4409e5701"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "4bccd0e158f345b34d583f929c0a74d2b502d3a8f4112aeb8a3e0e253a817c52caa947a62a4bf857d1e37b3fc9a5dead09fc2266319733ca2371051a4e358e0e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "efd8212e7ba7985699cbf81402e3f0aef704d3758db890f04946e436ec34ce9c900b4490f45a24c515fd6d91fe9faf93e07ec098656eb1bf422100dc0ca50f1b4c2ed5e6d516d40855224b122b07acfd782d501279b26a0714b23a6bb0c683444aca3ef6c5e4906682c1fa603122d036e798ecaa708ba30ee681f8aa829d105eb366e6896dcb4029c463e243b9f901855447fa2af302441ee117a3622a359fb3ae8b356675d3cc97428f6b826b922831e7c3e458a91e357d2cfba45b5093198964c0935784041cf925cdfde7eef72e83ca9310fc3be75e73f1b5615d9bc16429fd8d68224267199694a50038eb30f9c3223fe8e05065a84a55ce4430b3c66adba07a1c905903a9b8b8a0dc3e094373391a50d2cb91a1f94882cdb49b41aa11e6751017696ee0b5a5b945f9c1f13a832cd74d3884a5a23bb7a03a1968ed754394a5d6c2752c0726a6be1dbf894a1c10e684be81d4367054ddd69eda5d5e30e72b5855837b387fe203efaf975dbdf204ad472b1583e9142e034d8061433de3e5f4b513c3453d50d3047583e706f2c346ca481d5c596895af702d6815e131942537e59aeb99e183e0f17821184d9097be85b01cbe1e5bfbc16e9dde5e01e34d8e3c10726140cc1ff0111f1293f840775425a7aea399694f4b43c56ba469fa150a4a5c218f02a72cedad93ddbfcfa969f0f4545d73933b983438ce964fe21ff581aaf431dba61ab3be0dfc0a039c74787a38e0b096d479de6695e0e8c734faf3d9a5fe2aaff6ccba4b798121c92548a9da3a732f7424ca29c8a222e87527321afb81a39272b8664c7c394aab7833ce06f8423363754bd68242a7bd1ebf83706fcf860af38cb58f6e39948030778ead96bc974bd3021046f0bcebd9120e0f5bfc3dfe5cd465b5f02a6ad44b3630fe8e610f83e108813cc27c1eb79c4dcb949cc82e2fc0be65803187f73fd2470a4b1970d74c1e885c97f12d505331565cdadadb88625f38f4ddf98ec81e3eea6d3cf5cafe3b29ddf9e97c0b3b07beb52ba9be3e35e060ca7a95f045c5b8b6a8c19bcb776a982b8ec2fa0a1ec51b6cf39ec9f0f81beeecfc5f70bfb6d75763076f2e428005ab4b241d92446fd7d662971f01171124d72b570b360ee9e4416d91ddbe03ecb5bbb2267478d2055a81c929151bdb9237e14462e6af84e4d5bffbce7412b5c36412871c2e755be83464c831e8dde6823b5660a559a3cab1931935459cf4010ef9f6f38fb35187214fc78838ff10dde91a7ca6a9cad988580597447ebe2a8fd45cb01e066bd400b791f230c5c47c6bcfbb9c3ff5a21a65df12815001407d061d75882f09bec56bb2e42ce9619ee8363d06fd52124a2e0950a41e80578cb50ff12cc54e2014ee731b2fbfdb93f51de63ae3e9088628dd1db2a2b7fd07089a515969b1a5d4d515bef95af12de332e424cf3aa689dba6ef490ef394ef991ef3755c149259ac6903f4544c0b08d9bb039803f37dfc982b2b346b81fd7d95358a220315b195313880be34ef5493fef942e9def6b689461e5a87d1333acf84b425015554567fe9ab54c4a22a16f7d9546bbf682b995a6ce944e949f1920eee058db95ab9c93f5e2846ada5e5975bf33544c4aa7b6e9492f3b103d9a34f62598f8af31a2d51fe18181f21460821e8d202979eabcc7c6241a4697d8c31b7f24fbae4579e9637e49ee4a7ca8c7cfac500252af573080ceaff6433fe678413e757ee628ceddec41877f2a64d7f276a1c0504c5720b156654579bd92e7482aa68fa93cd3b0feb3dc32eace45527f2bb12670932bc7addd2c9ceb121b6ba14c69ebe11ed0b0da7ed0c487928888dc533d1b09a387cc8469871ee0ed43e4123adb12ed18aa5eef81b1c4fc03c8eb8891c8167219e5afe3aff733e1bba6d1295a8c8cf396cb0bac80e1acbebe610b024bc4915052dfd23a050f7463e9aec3454ff0b2bb25ad579fe0f2f4261efb67e805c5241aead0e3f5fd4278fe03d62af46faa3c9a51ebc348266833e8a1f27298443c47384e6345b61d029a06a6c788c55e1b831787e0cf33c5b3d46ce3d0cb917bbc634ef9f187d8d8f250142edd6d268fb71ec95436ebc143686e395130c37230a800a57a262b0cb58443bd28823896691f78162358999f23e859b4767d0aa8f791b8024b5b9f90372add5810708b939eb7278491b0691bebc63f94c02ff1a4350bf1bbea1c5621d268a57b6302c6bc851095ab0be8cac60f847820cac428f26d793658d2d2ff202bc004adbcd38a7fb0be4ad01e7c4f3c03a5ac0f02c41122256c0e5fa239fb1987c393557a4d373ba602cfca35c556dc6cac33a5e4ca3aa1e146d236b49bca8c8ad5da0927ea6d9ed9a09ccfee6e9235befab78bfd21160035ac3a849692fde7b999313f42258e07cffa2a77a4c251b65f33a037c87323bb770eafbfd499f5b97860417e55a92904fd820147f58b35fcd1c9762fb33af43239e09a4edb2b52805bdd833cdecfcf24f6caa9c75d03bf77cb3eb1da69e2f802f2c492c836b194cae8636f014a13d0d620569c5c47d64e959ff5dbde48f5bfbc461d11a263e72cf792846d52241feea71e9851ad427d73f4b4a16c91cd0dbbbc886caf63df169ba4ded326a5f94e68f656125200e953c013cda3ec9b82756c825e35b49324da514141b1f3734619e0624b57161d8913a3513f9fc451f5d742bc1c0ad996c1acfe5b5fb37c50772cd57982d92649be179e5b8ed78c6b93adeeb8665e77153208e3618d5a29b418d1c88a711b0b1c04ea21812fedbd3ee853c105a1932564f268b0e25f188414b7b33a33ba8f2d1ada05dd70b764df5260835a32726b6d2c8f66d75c49610408b04b9af3b3b814bcc9f51298a9bf5b3c98972aad4d29a6c2656dd73dd0e621f6bef57b58f1285a78413271678b7fc953fd19b30496607add10f8311589fa7665206b4bc1f75256e3cc386a6bb108923f0a8573a6e29241b32e906561d92079ea80f79ccd3ac89275e7817e23ceec5e645d54ed22b651a40afab0f4b142fa94ed4d34ded93feda5a51fa82d1e39bd865a42229ff686010a4b11f45f626bdb6fe401ad92847b5408f7d116e7b995ee928765590ca0d642fcda10b51613986553653acc484d717e24f4a8583e9447e20b824372a3902b79e45e538da9436d0af7f7550b70a2b6de921201d6a22cd44a69c1490a0572fa18694e76e899506e4ec600ed05ac12bfe7c8e67a2cb2e1df8a35cb02899297c6c0f400d74066e8f0a336286f254b1c014c41d87dd4d161be23e815fc6beb9d25e60d3c114f827bb89ac0b467f4ee2d0f5e956e89399e1ae49a0928834b70e17c1a2d56a1407c671ab80e6b43eef5d44ea12c1f28f5797c077f11795382b3035dbfccc6f3ddc5599b6d77839f9e029053b67e70e982e4138b25899ed9bc32812f597423a571fdb1af3d84b3e76c84efa30114fa275c6e6fd43fd9ff82426fd98569bbe322a7861deaaace6297e50d50a11388118efc88ef97209b11e9dfcb7ad482fc9bf7d8deecc237ad163d920c51f250306d6cedc411386a457c7ccef5f3b117bea491a098acca56b95554fdf2620b59b6546daa90833c9f51a5b7ba5fb29faa8a938c7923dc679a7b966a83500f4092826cedad8ea418125814270c6965f4bde8ef54dd010a2e855c5fdabcc5c8569d1d2d89810f026f75b15e474fb35c54c4422e6746d0a48389e4184224941ba1dbef0cbdafa21c873c05caa4eece2434503ec74030e34a1b1a9b662d357b87b6c18080f693e85103c7ce0fac840a4c17d427407c45d94939d9facff869cc8b0a007a20c0d8324e69cb6293da89f84273d2a43087d02350742a7a9c305c07941ffb2472a8d710733d997bddcb7b3885dcde3e0ddf7a51ce6eaea3716176cd317d2427ed1b3b135dd3c46263dbd21ba4b7594d3f0faed5316db8fdd608348df1860ded06945ebb6ce0debecca829422da7d66a5a84e82e1815dbf240912fba42df3afc45a0cc693ec94b095eacdb2c7f02c80dd6cba0dce0dbaa292798c2e2a3fd1dde20cd0ebd32119bbab046723b176cb0a03a6073622a4385fb118b9887a34438fb2e759027e54e334b6821990e855a22ea0bdfda4e62d211f094f60b29bf691edbd546eb29d6d3b443ccba9a17573f9123b1964002290bb0920844b53956b03208ab9d9a336536913c747dffd5fba18de5c2204f1dcb54f4680204014717de150a17f607e54f0b5c8de5e854dae1146bb8b40a5201e8424fdb18c089d361372a48b0fccebfc2b668b0a7861386a5fcdd151db0e34efe58457392f6b95ec14ff45cfcfe49138ecc686d1fd3d679c8ef347355b08d658c0b57ca27030a49c6ad05bf52972190f1b1bdebe55df814c850145c0285bae5ffd009374b921734dfc2d0d86843c2a8aa1aebe56e80f2f802c3ef578aacf630f84adc4357104aa07b25e8309480c6daf64c74c231f9719426bf47de7492b9e6d3d04396fffa2f1d274aae606dca88a884daeb0ac603ee6992a6905956e390683104d17a9cdc3eeb72a5b5e05aad8281d44aa137914cb3f8e51e7b6097fbf4458e6cf896c422e4cbdbcb7128f1966ef7432049d13a407cb27c8b4b7cbe686fff4a5d3b53fc6adb1ed12072b2b91188997fd05750176ba336e771831630956e06037a1c3aac106c64d1592d0627ab89b8e8ff2c4cbf4ab1e6b475d4c5a52f78fa38281dc359b0232e8aba22abb3d0cd05fce16b1fa85a435251ec92f362830b3c570bb2869bd699dc3bd7d61be40ddcd14a4b86515906b20a4e6a60382f2caf4cfe896cecd8d03b1e8346c9cf3ecf23db5f81f07a47fca86f0853aefc31a8efcc8d15bf6ea7498135ef6b703e3a915c648b194c02fb9860f12e5ce1c83074170b81553ca46dbef6e2deda6bf15e00c10d7b95fb83fa66a6411eca9b24921aefba79bc36f6b28d1685eff81f811f7a0c469f7c8165bae3d2df58415d89ed09c439e4bf610e1c506a9f7168fd56058998a075317030c39c611c69458b6097cd2f0a40d33d32e1856da0319197f8969bf09b8b593a6d9afceb98d03bb75128d19abc2c9c1e0dc8b89a387f0671e1494daea840585284a781dcd52bb863d416b391cff67e960738466ab8d83c1d98a40b1d4732b35b0e7fc7bf41ba301351f7efaea63cff37b796db69b1cba6fd872d40d708e10eb6739a4b0d6f2a37486332f24bdf6ea275878ad411a3d7db84454bab666dc9aef74c7d6831cdea3265705e47e3fc19ecc20f7accfd1f7b4e645b859745d65b2880019fbbe6ee88b60d11e06d116cbd42cfa7fd2286852473f0227190ffe32410033b87c277316436a1c05fee5ea1c758d0f97871386561e0ca429624ef04a2d9d342bbf87d472b48c5f42338264bd673887c7335ad30c219452449cfadbb660a82cfa2ea53904c40ec35e0c702bb5947589973ec3f97b392135022fdb697cc06842ae304b48d35c376fe0f376e80ce28c3d9010179dab4cf975ea84b940a637279fce2c3face765508b504f5e43cb3bc4cc39111d6aa543653984015c083bac2b6ae28c4a0b48c5a1413787eee3f986665e1aaa9f057007ada9590f8c6a4406becc73b7340eff6672b71c05ef719eabba1abc416319147f19eef017100dc82f26ed0815c55c122e0b1587302894c3910829a640334c9a92f130d17fb19bcf0eabf4290ec59e8c07b384007063f56e2298b3600dd546092a7c9685b037b9628b970a5e0afb7b72d8b42c102d663caa45b1df731211b0692e034993136a11878e30563071309cce8954343ecffa107ce29dd566b5b132c385ce71830f114b48ffc00126ab2ca8a701dcc413bee719eee2a2bec1607b731e192534de0bf60fadc68d32c920b748062a94061663f500072b09c20d7a391c82d150bd2b71df03bdcce0129f399924f2ebc320cf24ec6dcfa61a08d16bb1d4f7959632f2a895d280f700c9325c99e3732cb23ac66a93074f3b1cb9959aac44864d41bcbc18e022fe30377e0af2812d91d5aa7d849eb3778ed359fce271f11436e197820f29879c4060cb59ad6417ac55cff11090261464945417fa3e574aeb103ae1ad764354e74d6d0583a5182405c600cf1c726c7e776aa89fda34b47d711a3e666476ce0dab2d412d247a786fe3ed518ec9b3d6b95817b539c549f1534f5ed670d099327b92e9c28b63b2ebc2ecd0aa753edace8caea3652ce30774942de4eb451866d7664288da6669d7d8e3c879f4cba46fbbc122179ec25bf5a608b2b2632e66652696631270085e70a99b20f015d922fa051e5b7a4a41569bef7687d22873df1f85d29e906bb76906c5b29c3bf7e7e104136c02d450f2fb82b8a8544c205cad1b5de39b09b5bcb22094f0e364803c09144f72be2e981d73316764c0263531f8e117920185355f62467e1d9e739a22c1b320fa388f0a0ae5306673736a4e39e7fa19a5780cd3e7a101eb00746e2cdf3168eeca17ff8e007a081b9d162052bda3edcde4aaa7779e96d743620e5f1ebc82bb22ac04860c495a40f0a9514e2b0554429a193c5d09ca6a52ee0b134d5801dc342265a72d6c2bac604b61cbfec0592082e70ad93771b444fa54c32c41c1f6665c57c0a7fb96ef99f1968b2d37f90132341e80273b97a44d56de52014243e349ba2984adf972dc487a38f01e02be44b0e0a643ce7364e1adeef27f3958c50facd70178602de93ce18399584766f76cc30fe71b77e7beb58b4afa137d8b30c3b839456669a1e1558e94218b4f6b8cbf924749d0ca9e8f660ff35cd774e46a47956a5f146f5d437175363850e0196586461334296ea0bd3d63708290c76a5ce36af2b54a1b67f8d6bef046709f796b0806f810f00cf53182ac49d2091d26eedf5f57b1d215c9e7cc924bb3b43692848ffb8045eb4ba6d47d1507b6091f1e09829e23b973048bf79d19cee7e14d6c99614c323be3d4b98d76c3319557755c93154cdd37f490b3128f7a014bc28b1093c722965a9f4e3943b4e14fd6bcc6e876e0940e921e624b1fecde6c451e99855809a5350be9622be08095e9e300bac8982b9c3d32b9225a05fa6c983ebc1ed3d26ffc96b487b8994b22078823ebfa06e33e8cf27f8d78e9ae024d0c9f65c303e7415042bf42f78a6ce261b3fe442eaa3d608db6e1610aff014ba850cee1301c81c6c8bf92e2c7072b07c8f3dcd5feba90dbc70fc0fe83a4e61fa6ed35f80c2687478d97972fd21770db66cf25641040c3f56188842d89e045c46f6a3a72002ae10cc65134f1bdc223a463cf41ba0f43bef783a695b4c4bc06832a4f4716b32e775d9c76d3ef33102cabe77b999fa1cc692957f35b5bde3f12a4ff25c3adce3d3182378ddbf2da0a7338195c93db6ee2acd202f63fec7c724f6f177216fff85cf7c2be1eb4c25ab7e19321f998351f8d6f3baf37a72d495ba62c9ac1a7d4da95b1675ab2c5275c205c0328da11765cd59aad843ffce8ea04ff22b097e0f41bef47b732d078e9884bcdd156886466ccc3156ea995a7e30a75439a7f7e28276473a1fb6f0034cf85421d46d28afbcd185ec8c647c10ee74fb558dc32fde7345d27a77a38ca62a2fc0d32a8c5683c73e8af9bff864c9bbb61cb2830ef9d86e4ad15427eb6b7682a94b2f63f56ab9120e2bc77608dbdb529e9c29adedee46bfffd45584d104eaf2ff63b5055aeda619bb18f4230859cc3f78f9f952d295cff4df4bdd2306f179f79c9af2730f47d13adfb30e6b895722feefa6c0367f58daa953b87996ffb9ba0cfb6176519511d111e69a54940e6c560d6d331cbd26dfed5a179ab9452db611bfb657af158ed41c2b6a2f8550fa020be77c1f4d5b6b637e637572faa2138f276c37de0b335ef718e3c4f31f06fea08b1986dbf095dfbc047207d1dc52339d41bf0a75293f87f35dd7a175604a6ec07a1037e994f0e62225e24927ccca10791704a8ad679e8c7161736a37178fdcfa4f67e9d8b8f1013e4950a3f55a6e3c9b1df98cf774d4eec9ba4399a8934bcbb6818d2729bb1ac378e539c2baf3d31a8bacfa41d3af772194ded4ba5a7202bd7d970c7a4d39d2af6a8ffa0fe9278d3cc36a8341d295f9c74b258336cd224924c816424dc3b3bd277bb03df62073f105cacd111764b6e1425544ebb9612283841e85db3c1da74ebe34ec621fd147a3281a5b403d1a80a2155bb38a9cc6f1466df94c43afcb82ebf61eab02d77c4117f62a8b8d1d6491c0d2c6d7cc6b5f2d0644e0fff8234dd52941a1629cd791ced40283d45452c358d1ed1f484dafaa2447ab5e178531252c9b3365a99c7ab26f4bcfaecc3257c74439d6dd8b6ad428859f03c9095743e1e7964dc3e68e4b4e90f3d013e9a3a6d0f28d2279f7c0636aaf50d0da6c0af5f494cef344bfeec3d6d270964ba334ca80d230428d33daf7d12011a4dab60ca1ddc299b6708c4a7edbb22694ce9f423d4761b7c97fb606ec0c8665f2725dc8e22def7b19dfe20dd6ddaeaa5e15146869a53fa839da04f0ca7d20a24c19e9ec31357a079fdb37b5564d265054a7c923ea2aba75f62497151d93ad7b1439d0d319b4f0b3bb0b13ed87bac7dd1b0f23e1493860a580d7e2e04ff08b44dd6ab919f60cf203fde80cf2e5a6f19503a39b4cd16ec0bb8c7c8eeefc0428842a24f92d70adec7ccada6981fa82320cb11db5a34d2e692c5b7ad19d8c5f23dd6ce11789f4285e82c48adff730dd07c8a5c609a10707b87c906b71788b61fba528279acc0b0cb6ba5d51d6ef1337f14d5444edb2897b4d20adb6e51c73ad2e7fee8109402fbb838807d7c4c62bf38d0f4f12f8445f0975bd15f592d5cf912ac21255d992ccb5438056754018e5c01982e8a8aff2f222f0b65b90014119ce7f10464e82308caa8517bbc2cf53db0e454cd38e8b3aed681c81012eb6faccb599ef0cb3a9b5c9ebb6c2ca38d903763e8ec1722d0088ab4977b1892a3a4f62d719b2b82da67f1ef95725e45fe3a431e15485571ca668a63780dc660af1db76b36d0fdb250f18f663f495ac9075b2b897451ce4c5e35788c2c6462a59740b72cb126b08325a0fdaac0db6858305bb54b0cedccdc15dd1ad6bc6a3488e477bd19e8838b1ce0db0564200ecf1a02a2aa0ce605954876c638ea2f0583321d2993dae5d1d5a6090c6d3c68ccd9360498d176088272b90a599c6f8d1f99cbcabdff16b0f6d748299afed99ffcccd5bfa5f42f7c4aab5757fc0d0ede9fbc7a739dca754fdcd92bc7ed80e4cd86f11eb932d9df4a478849e718526f2b6d3567a130bca1cc6922766b6dbb04fa51903a7ca5492f785f51b8421634397b714cbbdbde9c3f2e87fc529d8277e22f47f15090680cc6107c4dcbee5c7b15d793489eaa0f34fa7e1dade7e6947b26744f5b897be1ea4e1985c5ab7554efaa46bb7337952b4a77c3127e5ccd0b9077cf51180dc77aab6c9c"))
                    (acl2::hexstring=>ubyte8s "2d97d18d0b70ab231a0ecebed1a686c739914673521ed64ec6e27c1c78e2000981df932439a697743cded208f736164468a1a53cabe2b65b9a0ef10ad42f0372"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "0c25d4a35196e3771cff30a7452ab30e68d71726cf0ab02aa016432b1d643dd975e0702f458086d8ef7a35b91d9faa6d29e7458840e1acb02e208b3d79ece077"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "7b87a9fb861a84bafcc146831ed9a54fe3512dad25668512b17339175b0c9523596209336b0ed0cde5648faf6c101043ba3a653bde0ab8a8c3959b33aa9f8624"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "afbe632820e278b633260ee86ea0c2a88735afece12932ef44d5e762168bc28530a88bf49bd80ba36ceabafbeba7fea0ba783398109b10bf1373a3316e72ad68"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "7a57070de9006a0eaba8e38cb24c0d088e08261a86dac157e8a9ba779d866fb800312d7df1dfe93b243a3195bfaa62c625d31a400c5ff092d6fd638f1ea911ad912f2aabffea2377b1d2af4efeb6eb2519c5d8482d530f41acdab0fbe43f9c27d357e4df3caa8189fa7745ff95f811ed13e6497a1040852a1149890216d078ee6eb34461cfa6693ba631dbefacf83ce5ba3f531ddeadba16ae50d6eedce20cca0b4b3278e16644535e0859676c3fd5d6b7d7df7bbe2316cc2bfa7f055fffc2835225976d9a737b9ac905a7affc544288b1b7d6dad92901162f4c6d90c7ae01e0ea4b5988215474d8902fad0f85bfedc351e51e835747e3ba625086d63c7832aa7cc562119f9199d55a8f54cf2f093c4f3f6ba1be5f0524816d0db321e5e3bba9af9998503bb35158d02b823621b999f3225145d219ec2c33f88bddb7ea7905a52cd1e58d1e41032b9a0d208d84576b78a29df409ae836398aad3eae174c278669a75300f6fc319145f3d171c7f6b1aa7685995a9465232772cecc4e7ab457cb3a28b611f57c2fadcc7c14d2b3b7e0e461c8c10fb08e2ff9a25de5111b68a7a56ec1d5f05fec50e4b7ca3ac02e017ae1d663f4d471dba33c73eb27ba7bd19d8151f3214098b54bd2823eb1be995bf64f6ec7933d5ad6337c71f6f28f0ea7b00b9e3d3677327f76f3d679f812782fd98975942acecdf6372626f1d42356337580ae207408af9378c50edc9c805d059b0c953a2a0d5db444b5fd7c5b0be50d54acfc1d687972a9b7c2a6070d7cba75b155482a2f5d9ca8d2c88fe6d5ad3dadd3dd22b4824e2e1aca793669d96c12e8c2f009049943ed4c77ca38d2028bc29d899d9c66600b347f07652a757ed2e8e73669522a0654c2079a2792be302c0644ec5ad3a37015284f14f31cda6c45cee9923dae3ad504dd72b08d281c0d029b3c52b1630c31006bf379bf67d9203917784cc98787a3e872066157636be7a3d3fc732db501316192048c36a6124921651becb978c0f154324b0312cf3e5e84530db42c6412ef9d1d0ae2d5521635933a3cf869fc572bfc90a94c4bd603e0388ae8ac39dac1ca377f7f1437f6558412f953d6778cbad2261fb9736f224d0b150e05a1d2938e6fa52bc976cafd40930545192d4e947c828531f43f091e8a4dbe3eed5b9328e93e8ac482a0faf16800fe8fc5537bce1c9b8e69d1f586253c06a8a0183fd80552b54cbc3329645d4c5c734c10ff7b740bab9649eb3d72f0e2e448d41eb52da36b725f85435683ab6de3e80c6477ca30da215134988d6bec0f65334d5952109d93e457b23bded5563d55e1433852e47ef8ab33cc725bbd8121e7dc7e509d0d6e5c89b9c59518d6084b20e373152756d8f797b263f12ef45acb0ac8d9bd0c906035e0fa31e7ab021c07b3532e834b9db9477551bdad8dbcd35b3c7cf1d717264da0689d799bebe62c03abd4d7fb5625aa78d2961c0fee6385630b6d5937435b713caf182f3a81fe92bf68dd01e5dd8b270c1df41180c774db62e61e7266279c655b22fea53bd3b7eaab88d03efb383ab961d5f030a8bf8d5feb5b04dfdc1a55a49c52c57e2877bebe587a52e173fcc0dc2a991efc002d099d5ce8b4e421c233b960bce13c88d45f7836bbd98c189199f9b6c5e4d5ff9675ee47fda341af85a31e58cfe652c5702692d6cbd8a131ad32aca6eac6412f825efd2d28545afdfb204de51a594366e1b0a8a938d4c35796376f6b7657dff88721348cf74ca2958937b2861a9e91059f6a243e71626502e2f4bc8506fe04e30b691ee7d8f0ab0ec65606cc4f4acdfa0fdf9ff703e608cc2a171e840b4d5287959326be4bf4f6a21907951836c25cecc97d66737ee4110bc20affd8f1aa4b5eac3be680ceadf82827c8f80a5fb9b02642b0222430f1205a413cfaf458b941c0b5cfd8cfef9031e949269a0328858a6f02346d99d871c9203e497a86beeee86648db680c1d50367e138d15a33faa2e0f8e668344b40a4e05ffad08d092afb1a7ebcb3eb5d642784f0d78d56af3bc2d805dc30da5720f51743bd420925290001e053ba76db90b79d49d8663bae3200b89e6d6874c811565e3495b4c2c9d2486212b894c94240fadc75a124b238c7a3a54c01d6a9a7fa0eddbceaef60fc0d1a669d55098b9a5c3967fb9c8f6256e2f5c996004422fbc266bd6224bc8cc7c2bd340890e064ebcbbc6663306ef40f6b3c4c04d27887cc53a626aeabe1ee30d97af6dbd3e3fe3e77cb0236c48b0f6c4ac1561edf3a7f2394163f29e8262d423b99d19f85d8fe9a47ae62fac2ac5c7368cc0d6d9054b2076dfd315a4155f102ce62efdb0d880f0823acb03a919a822eaba49c784b8cc980c6a523ca1f34e47a7b1c7817d5bbd78ff30f117b991d2aa5fc9329c779ac6c9687543effeb8dacb12b1194db4438fe379f1daf4742fcfcdf44dfdde3837a42529953aa830b26f1afc48769b0c2c92d08101fb225f4019fe469d1df53cfbff4bbe2ea8edee9c75c7d2fc7221db7340c0b2c5c429e17f2bcbc55b1589fc74229d94b601296eb3a94225519e5ab8b424247dc1e85391398fa70eec76f50d2cd8ab5bcff60357b4642a61dc4dae0a7bb78c409aba67e50bb02061e5a4ded5726bf561e0d9d4300b20e126e122b477167558b994dcbf537237b89cbb419fec0166fc557d3f446e2479f6d3249fd65520e82a376491d272e970efedf1bfc0c7465420851998bc888cd9ebf50b27bc4b377a61486eac39b9ee1636f41d81e9edbdbe229f2a78a505c0cb210b095caebfa6b6ae7b377e5d9f0c721e3d933a63c5d40f42f24441a67c5cb84693a3a156291b27623e337ad6a8a12f1638f73bc6e625b2323ceb963932bd4bcd1d6e882b2de1471d0007341d5d68a7af64be28abd52f5140cd84f8c5ff6ed01516ce939624b5903f7f174c9a23af665a4f5c635b095d672f2294bbd508fdffbf0e48363fa3e48d59f9e7f6e5a201f59757568e8d093c6e3deabf820f05c7153b626ff280221aa8fd0efdf5784efceff72b310c9a3b58d0fd6459ddac6c66db1ab1aa5af035b47d992ccdaaea80fc9ffa2980600b361f2a0e5df81165a3699bcd8a9bc218554e759ee1dba984ee8d8000e0a91b68800bf3b1062edfa3512bf1d44b98ea50e7452baebcac53fffe2c5aabcf2fa13a8fbba8a61b429aedec3a57cf733616428234fa292bb44cb426557a6b6398a3e9db8ac35dfbe7d5bddb2147ffa007e865ac9a8baead97e64a0d9559c04c58660efa15ce492683129bb46d3d154062d322cce23a9fb0abd30a0a68833ac8039a0241d943e839b8c3fffe6ac180d2fb590279a8917cd37475aca34b814b851ea27c6931aebb93dd7c631fc65fd2e6260c9d295a1ae69496608a6ebace72f1fe01c5d6db7943c375be5daefc727be877405b6283513d6265f63bd342e531ad464a86f1233e115725dc581279a3fc5853a6b59b90bf11cb87d92967504008057bef386ea56794bd5288dae0ca6ec4c46fa2445b2edee44e507bdb917f79769903624295de805e6470366e906b2d759b1fc570279a1b25497440eaf83cfc0de671464e1d3b02cefb23578bb7d88a7f7ae872d1678efe657b57dabff205422e5c49fda34b477976dfd07eaabd9ee052b25e4ce0b15a1325227fbbc31554e66898f20afac6805ed6810c1e54282ad99fffe5c6169ba7d3d4fd5036ace8c8433cfb0df0ad15722d7e9d3ab9006e08d0a7a21350fb6ce4db055a7e0dc5d2cbf32c10f24142aea96fdf62e0c14815b5501a477b7ff7cb9d7326d85d83c669dc70b669537f43f8728069a092e983b5ccf2586e62c5a42e2e46c9e5735af71466465c6285fd39898a29bda4c5745bb49c11f2757da4791b33b06f0053d5f07915ee8eab1e6d9be83ca63c0dfd043898cf49690b30aeff093a6a76b773974afe57c3dd919122d38b2690753aae1b92d8b3033c7e00579e7eaebf718957016f2aa870c9b462cf88e8b462e9b95d1cf2fb03c9ed6c8100efe21d3a0fe482c6b574948fc39227b6149782b81a189eaaf1862e27ec45176d6be9e26472576b39e662884b6ba944991f0dfdea76c6c727e7008e87b9a657df8aecaac9796084fd5a7b006e22683dcb00b2a28490428b267bd0c173e5feb6f9d570d2c3558e74538a349952786d97ee522c0d10f72a3f29089d5376f1bcb92ec3fd070cc73f9e0f1e15cf82de17d7458be09a1a0bdc77a215bcb10b6d7ed2f0ef0793e02f0286725e94ff89d3f570932666d60a5ed7a9143bc5669b99a398d2af2725f7fd65b21ff448c5d1fa84f849ba354d5c50ba6801d33de0c236c97f669cb219bec2d7a57a47979ef5a254d2cbafc26c9a91e3290d948cbbf9a55efb0f305f0b9f1fe6c523f108075f7bd394bb8f94b04f92e06069c83854bc65e2904ef27468335501b57f69208b99d79fb27bc222c4e5c440a2d916d7478518e2d3a85ba8abf612b6eb1f1867881f55f7b08ef002f6f7a66264cdd8d010de0eee39adf4526361c974da659743a96a77aa238791abd2efc9f42aaab6b92d41e84336c225081b00f9da08056e4b5c05acf56fe81f83f548af4469021909193b05a06f7796403f5e2444f850f5f51c6b66880a12f231ba9cb6f1a4667836413417842840a0814cc3c9543fead6c7672654c096aac22bbf56817a614a6cc751c77e4d15bfdc41cfec596623257948c8a0c83d7a1207d850008f4d34206f42a76faa899c47b2083542651956b365efbd75616d48eb3bde392540f8717ae82ede873cc8407aca5b19307cec5ef72522c21e80ce6788b693c72f46fabdcc7fe253b8a56796bb7da0f860f14d80832f885306d6df7a9d309f7698f1c28da78f13937e6505f39c44703f4e72bb8fb9ed96fcb8b41ee67b1267e3fcd1fe456eae293a430c120bafd75f6a4371b4b13e07c53bcea91b3fb4d4738bc587fc99b44599a4a0fb2a33388eb515fa6ec7b971d6997c933642eadd0f48da7d544bef0afdeb6f12c2ad0ef8d317aeddca72ece8029a5bb0ec347c462b5e5caef77aa9c53fa4f602e318f3535d0db1682544e012e43e35922b5e68a2a13a8eb4c1f77e85734cd60becfc84ca74fdaa240f9c279523fbb6d1842f34551953ed394440c1ce41599b9840d637d0c36e158bda0727b12a694bd9772563efec4104a70ee51802124193327f248108c0e5701a250465296ffd513484ef7bd5a26841888262e5b37e9b6955087604d4e5ea4659612264507d2e835d3ddbe098d45030483f38bf74decd8a9a6d3917b7bae2f4c42edfa9b8da834ebdf28946cbea9f6d7095e509a8cb6e835d2dec4e43a87627e4de9aa9d4f0721ff14355f2c16d8c359c46e658d1cce4c027bb0e2ceb46fa4c2600e1e93767a17371c0e0992bc5cd452592bb21824bc6ed84a52d18211f05ee223653fc529867e5d9cbb78e6d8bd8f8a739d2082cc09b57fe7cb7b487892f6a43df5f37a91b5be373f42967094815ec9b8cd95153e3c88bef923c66711dcd5d157c623fbd68e3f8eaaad49771632a8d6d0c2fa179fdd515e28629c290556633118e1b8911f996a2988047b458657d47a648438d11a7477281e91e4d47c88088b294d9882945fa5d27f4d13200d794c7d1c1c358bd48a62e180b886b19dfb2da83112ad165f67594f3792f951d31f77a79f91db8df8a67954f9dd690891363851d939dd97a539ef695ada1520ffd1bdbf96eb111eac5126482f0a772180c1934cc9c9124b915ecde88d2e4fd4b918868fa833be871812d184ca4697439125a461a8ca337c485d5d947a958fb72b7a3add7e70cc116671e0d1739ab1b8ece9d55a1f452870f5be0635c917bd0f7a814172bfaa2201bbf6bcf7d20caec6ba958dfed540d96fbe0a2f9222bcc125d1b5041614e9a8a2422acdbe8ea774b623b2331473c62ec3cbc842742b8d8341569de9902536463b20947bee4f70ecd04ec73165a0dc0fb34580a6a38f91ebab31c159aa9998e41380bd2f4ebf71fe5b645ceeb9058c648654bbc3e7f4d0e2aa8ba9bd2a19186a73882b4fa784b193a18cf9d2d749dd15851fff010803534e878cde7bd94cbdc0a3606845c1463891e8ef27458915d80af8cfd7c8cd6321198b5be7fd7fdb8fcfeb31fc7764ae0fe279b46daf9511832b278bc880ffbc4229fb214ca9e46d967d93236d4f58633d6c9c045478c4e4e861597c5c9166b8bf68220596e9cfabdbb3e25d4533640eca1998076308de370f92d7ca1a63cafd9fbc47304c240a3c655a21080b8d5257bfcfb7e422c8abb1fea7ea9c8a5ef333ea0fb1c9f6c15bca3c58994b48c4e8e523783fc4191ef3fbbf89d81a35eba903a948a9f91c2a8bfa69978f4f8340ef61fefb79ced1737dbc5a1ee96ea270b315e34f77e847c66276f3fd4e3b44692e725840ad2bd130129dd55067bd5957da10a00c63bdfefda7e6e2741d3814a76fc13a2f2302a1bb7a463a069832e306e69ce867b5a1ca3724f13fe3436e796278f6abb0227cca7e7d5e9128c6f2c7e79f62c448b4bb32dc9f4011310904fd0212b8b4abf3c1d3dfe28ee09d37288d60cc0a3b97a934be54e17b6048b72328ef05af5c08dff84448737d314f33fabedb373c563cdc33f608b249340f95f281911984abaf2c4aa620f6d38b250490e1bfa69f1c1f95e1635ed952966eea9a905116dfa98822de7fcf343af707f1137e74e2db95281c658cbf203f8d4d0042dd95c3675f7c0bdd5abdf5587bdd41ab1e62b399c940b8e8f9944db822941d13f1c6e9c5d4712d6ebc5201d5228f3d7f48f1e081c14db392334329645c2b8a35534d50a11050d2ba2ff61f5af1062db88489333a8438538ac1367cec43fb8fcf8b90c0a11c0240e83a11198ff228597d5a92726d1aeae1f92bd30ef7dfea730922f90170ff6ebf21660bb51b2cde08f7537466c18c4aa406fa84a62e39e1800c9fefdab4c17b998d394cbca7a711834ef697d7dfe13fe6dab5b00e3958c9fc008ad915166b1314533d3cf5da6ca713a9cb638d6e21b3ed9ce4be0cbf4488300d6c21783d041d7a9cfa432a5d828afe305b414f17548d03751018696806086459cb2eccd29480118ff7ca21d7f1a08c0d2d97e68283b56e8dc025ecdea52fe00f3268d87c24262255bc37a3f1c2d30765cace1b56b612760b7c4d5fc629e7eb55263766e4baf6f415d1f47853fc09b563b84fb43bf38e6518ccf888cb9b17ce5331033cb161d8d1f8bf6768a4a72ce02c818179db441081db398e319302a7d233da34238c1eb4ff260dfa3b8ccc23ceebaccb34dbb081147d4a7720c6667e3c1aa932e31dfb72c7c64314b8b571d6c7c6c297abe63fc76425881c1e0dcf8657322766440bfffb7100e91ca9c4a12d15e54a4ca7a2facbbafc79e3f21046a66aa3db360eff323545fe887cd4646110ff71cc51f0b7e8755d2aa7356676d2c6bc954e0b08a09d098be5a79b013bbb67cc92d1104ece12ae97d69a7b40e77cce962f064efc97475f8ae0cfe1faf1b61ac56fe0f0898f7e5cae28d81b490d4ba740998d65b08b39208b1487b6862f5a113231815968a13668eb49eb7f498d473702fc6bbaec257932d6ac57826c9db0355bff3cb3b5ad42bb96615b0d08127867248026b37a0d117fb3cec7ad75e2bf6c87029a67365aa83cf797ae2f4d42e720ed4c48ba21ea08ee6aa3609f69a6909fff6222dbb45172d255146e4ce1c59b48a7895936a86467664252a9c1109329b70c465d78f639a867fa0022cb8d449cc8985edf8fd82435d184032b79314f949af4d6d1a6c6bb50e22c8253161767f38908f4642c79d27a1679e06c309682d932ff3c08ceea7f742a34c2889447d5a77241726f7753163f1d660ef64f05d5fe65779d8a562b2c1b295ae949ade737c06a8a0189d72d968bce96b8513c47ef0d1da224dda8516a4ec6e51f1fc9bd34475b2ac938b4d0fd661150040361247a0d41caf68b2a24e314e91f8b90509375432c6af1077705f1d8cedb1a9625631b07b9353e1c466c3c757a4e5626aa2fa1bcc592a27b86921aeaed802392efce7c026d2b221663f0207962bba769405961ba0eea031e4a3f22a35687477d292b450645f6ca03e879352987aea7e078db5d46efb93d772bd6ae0cf138ade13a431e1c29dd5f6a3fcf238cd26e7e13bb7c3c675049770e921d692ba7da7e458adead4a012709d4cfba93aa2b45538a7203a72619e0da6ce5def80a5eb7e13281c49d9ab51defc800022d8c7b794ece0339458932c78e188ddf06b1ff15ff2f87bc94142d073ace3a6e9839a8539966d8e6bf9b8346ba9af3e93926733d00c4048f3abf512d116ee5c77db99c1b59da57d2bc61f4a4867b7ec33973a0770b79600539a30ad69d3169d583153d19b9b3eac4e5e56e2b1b429537afa0700b48d703e636c83e4b069e2f6053547595df78f885e22ed721a177a20b43b3ca40e06555fadc37d6d5733e9d30947e2bd6f3b1e9de5a4419cf3fa30c089cb6955ffbabbf09d0f7bbba0c971c0cef9c47251ee62494f5301a7d1e4d3bf663b0bbb0c032a7dabd10d342fb3315be0cb5b0ca87a013725cf9c47366fd979d3faa237b58817636498b08b7659aee0c92d1500d58661d3f37a7fdf83e3b88c4d9abb508eb2195e51c6f415f755211bf2af80e35b6c1aac22d43af55ce907635d8bb599ea0986aafdfc4668450e5e49f7b33d28ccb3822c4e7d3c1d33f83c9b6e5e2f7879f705f5c3a925e6c21dbd1732e898458bf53ee0d0d621bb8b99de910a82977091130dbb3d4e2135f0eba3e05ea030de2ba9a746546f5a54c63643e0bc897c16d1e61467d56bb6ca73f0c951b58a85372e3961d943a9647d96662316eee35befbed8a07826726518bdd583d7e1f803f5062ce5161924ce9a5ea1f2803c10a54faa733888ab02aaa5210279ef6e893b9c360ca7996f62073f916228c90ea1e6cb473e5305dd9cf5f45ff00204a5beaaa140513fbf90d18cd1d12f3c6de1079ef6f7a2667c2891d59acd36b2b441f8cf88c78304153803fdf59071a3cb5dc24c374e3819583681d7ecfb3f027aadd818d71e2a9b16bf69a92f51d1f58dcf626d20bd77b22661fd6b7f5c8187c453cd24f0b6a07b50b1833cc387b8b811d0587027b7e9e77345650ed673a6a92dde58628a439d1b4f98e11766c4ae61e3eb14f3d0eff71fab4b494066e306fd23a7fa5ff82768f16feff3c79d7b1b69908a4876a0a837c14b0e2a9ec50d2ada223b338bbee4aca42e214bdbbc1d9855943833771b831482dee8897855549ae6b6129454630dd08a26d749b2674e9dac6b7f9653a60325cae384d67bbc7bdfa96bd88d7879056d49d67c7dea1370a3e482bac1bac4aad40fd54d92074ec551789fc79f36656ddb39b97b78f1351643847694460b1bdcb7d840ad5ca9a0e3a8faee9a3a3cbc99c2f04c6eab4660b1e502b376a3d3003b49aaa24e96e147394ea6c4bafee9e32b8cf41bc085881ad610e5c5ec6b812628f1fd668bc13ca638aae2ae944ddc7d95c1bb841ba804c7f54fb43a6ffdc3d23849572a0f7c1a4284465528c0b762f06ae9cd21da836afc8209e46d6bdae9a8dbd83e59241808a09eb964bfcabc3bc01d6d9f717798192d2ec7fceb8aa3811ea5f7197074359644da7ca7d01cdcc5711fd5b90f9d56adde3e12744126c818b8095e7f0c6a5531a3d9f8850631998e9804ba8038966f77844271d0eadab91ee99856e58990e779421fc582ce9362f5894e74aba90d1324b3de5e996511071247b578d374626ccbb41e5c2795dcd998e96fac91bd5c1233f15431442c54228136e34cdf596965a75e4834f9f8585de71cdabab9022dcd13a166a257cd3eabea5bc44b5fa510629ab53fe8606290707eb83237a803dca44258682a4ab41b03cc0abaecdae3f1d4246d77df01613fb1ce3fe89bd53ef3484e9aeb21a36d83ba94e859b9d7a2784ac2cb62813380a0723408806fd7000c29a6cd6230e8760340d"))
                    (acl2::hexstring=>ubyte8s "6be720da8b47b9ff169100d51bc971bc9cf9968cfd305c6bc4b34f9fb233b3dfd4dd0eb59def8f329fb60a32be902dc8aa3aeeb8dd7c1896d0e0bc30ac1a007e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "2e232557233b61c0aba3bf10b0cbbfb953974c8931e639b6efafbac76320ddd47100ab406ce17e216463658ec43bf0ec8acc2bc3d076ce08283477c5f6149ff2"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "3844eff6b7cc4eb3daab90181ccb8947842e8c702858e59089f96dd95600b866ae647f9cd9213d726b4fca636e6166a2088e2be0c27154c28732faf360e30e8a"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "6064b25061c48d401846b922a6f28cf06001e0e9210a759b66b4cc6315b18a0bac5667e3edd91563ea3dc002c732d1b93857ffbee08e12939971704348edad9e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "929548adf1febc8414d7aa90ac20dffd8090094f57e4bf541878c256301483943408e5f4c77b43f2bf00b5dc836a8f41b7e22ba71c36e97cb174734f1fb8464053426eb56dcb904fd9192e1dbbd2270f918e7dad142f5b085e7557f152cdf46a396a6b5aa997ab85ab4c9eba1cffae4e54bc88ca06781fe167a3f4a593fc96fa2ca9644a44879a7a7dbf8c1d6e9a2ce84996d266bbb93ded1425d5d1a8ed32d77527e2ed06426718979a80ad794aaac4b841e5eafc99bb16ad247fdf5a47d3eb5c0b6cabb6711a45400602d205b82ecae9e849bc8fc0a34379c77c3571b27e1d287e8bec1ebbb9ebc12b9bf4d98bc3e223b184463cd7fd5cc137523db5d83d5523a7c61804c94ef16230d27bbe6c6ef4b0c420efcd86cf48cd9b8c5dee5e177b93afa5973142e03f6b3d30d07c0339e8d64aa51e087423f1e51cafffe3ef1578c1bdb0d16dcf78bd247c3c59dbf72ba6de6b993db74a003e51e2458a5d313d32f5fa702f7f181d53d5137e7fa3f14c17104e86b3a0af6f175814d66624cc9992778d6c1731c4d7e9c52a7eaa9b98a521b31cf3e88c4c766444dae8fa006931808e9f2cf2497cae673deac5438c953dad11fc694442c9200d5b5d05b0b9b00be58358392990e7abb5857adab03e8f1da470176f2f8ccd72fbe6c5b4ef164831a79edf1a22ceaed5bbe503455b792f4bec19958d11300076142c3a7e20e0daabb0275d5747cb783562a5ad8faa051dd16f549d4b52f70b334177e3e9a9df0d7fe666f979ff4eeed1900c9f3f07a900b7136857325aa996b16b0de87ff3db8ab733a3dc07b121b8b32e6302ddd92e8003c29fecc9910e48f9783772c8669cf85b7e252d89c70a7de325f2fdf1885d46bf633f849860edca12523da94b6ae90071af351588a8d2e5675b01b9047cfcfaf2de9146282d10c304df67b490688c4a03ad1bbc8bbc419a49bad85e9d1cd344a51bdd5b00613ddcd809fdbb1fc64b7d18621f491a6b1e419129bf1345d2accadf016beba9653dbd95b646e881eeef41b9b589e5bd1d3385458ffd0083064d37a87a821da9a0d80d43d08b613d464040ed8ea0b7fd3a460b6fd6db4edf1e7f31086e6b198a79d575eb3e144edb38051fbcd5ade8612a207e16e5a125e1830a6008c36f08d35afe9c93e829d8e32b08892e72f89f7255898443a497ac129db66eaf62d4120daa192b46b61227de870f244c4ec905b62448c5fdfb040cf9d44b000633e25ebfd0efc90990166254b3e3981d4f6842b7fd27eb4cd2326c1f9190c1131ee3157136481fe65e7fb1cb194c8ea95e1c7dfd49ec9b381167f50a3a4e56e00c1d5eac43ecd4e408933dcd2011aef9642fde0e71607fcaa0eccf44394d6a75e3d962f65fa2910b769b2ee130fc9f1447e78e74db4b3f0967092155f455a3d88d5f48daf233fe6491deac90b1d30a02f18db5b7f17583fe4f0513c3d52ea913f0f739dd6468c2f63fa6de600b5f96cc48687a5d1aa5f2af3e2acf18429f252ef6e95d891e0786ca6939ecdba768ec7936c206f0bda534261dbe7adba572bf2f867cdb6586d8f1e6f8e0ff89ebb6b311a2e181f8443bf26bd50a3f95656d1e2087fadcf905a5ac54d2e33ff14010949d730e12fb1630d4844c5644cdea59eb08e3a987ce04345e1ce18ad39b0f3f37249391e1839b4b817b6ceab10b9cfdcdf081d2056be94eb0440b1de30fbe0c43aee9b217173b56a2581c0dafd8e3236ddf49bc7d581dbb4ad63ae399d6b73c61445db9dc405e17de21430866ba279172dd23b1181e0943d3d955251b77609a38ecd7ed57846c58dee0cb23fd3bd01085f3f7f7fe4414aef456442b917a23ed71b03f98b16ecddf499c9ba69ce346684e27add75652a5c35454ef3d246867fcb3987d001183e974c85e9bd96d39f16174832d9948d1264b9cdb3ea3fbc79390a7ab7a6da04d29e1032db30495c53ac3a00f5f85fe48fe0f73f419a440221171f4f11c298653f3eda111cb6e3f0f875579d22ce39d3f95d482a5688a18fe486743724c7bea1db05f4b12cb6091a041e0a8d47afa503e536570f8139bade5a03a3c37f3aaf434c900a8399b22b3ffd12cd41d33133dcea089e6f50220275e81b18647f7ffce1fa498618d24fcceb49d01a6b63d768c76e4078ca31d60aef25396caaa42c29d66abef1d164c360a480f27c6a6826b66bf87a8090a4da568277eafaf69ba3b8c9245cc85ddfc178e428e65dbf5e83af330f1340632783580886ac5de8b49b1c02ad9f840e91de8832a81858dab592a5a518a732a2487591c9a809ffe18d66e64b6d13cf74395f21089524d86d1bc84c4aabb2c1ee7635965e72607b81948b89ebe945e52f14e7a5ee568866b6ff89158c4ea298983f27d82f6c5c3b49589d896cb8a26f8fd2ecb95c92bf8e78bf7963f26813014011e0d9b4fc69c9994801676641b5e926e1f2cdcba036aa02fa0dea058e60a6768ce50da7b8f5a101e15b96312afccc74d08e45b4258a53e62dbb0a4e3837f29c0ea8f65018ac5c03c300b875cae3a40a1d03c3be892726a2d9053724ebe560211591264691ea452f5b9520aca806ea572617ca4341997cc96a710342101599c640521003cb2c16d5298d714e48d309dc9a2ecbdaf56dc0ba8092f0fe5b34e9cc0d5046000b0e2ba59b6e2cb333e4ad0ad7bd3afe0af0bcf689cf82d213bfead2eb102ae407fd0e014f9ba5a3ef982e726d47ba1508deac2e049b2b65d033175b581650f818753888b4aa9e78d27a28a3c952abc8559438a17f9a7d22172ae9c5d2d73ad4d30325ac592f59ba9317060f8a729280979dccf15921da7f6f03fcb2e9e75b02be237fdcab7e79d197db51261f0c00c64577d3ae3fd5d6063e5aaaa31bab240a405281aa2a3c716657538477f5936901c59f0728823af23f1b9b8e06d6a74833f01ce58563cdd2d1680c3a85bcad5debbe9fc200ceb5a1826d53116ea9701a4843ef160ec6c700bb3051cddfc2ce31bc96e68ec783ca8698d9fddf3b127a3c9fb2559d96f19cad7c46e0c8d6b65a972cc50fd7dd300df3bab808420c9049f1b2d1ccd6ce42d31bad2225b60ae5f6f094ef6837c251e48b90ae1ed443a1440c1a18c17c1558feedc6489d100cf007d08d6cac290767e2b083128e1080d41cde3dfdefd7fc9274964020b3612d5321a98ed338162cf08d053a8bf281449718dab7078484f49dd897a34105141c9dc8fe119998a9a37a47ec0f80c8a0ff68ec93fdbc4c0d8dc99f8488300eb32b2e6250ae564a3dfb73a7f77a879cfa11d7fcac7a8282cc38a43dcf37643cc909837213bd6fd95d956b219a1406cbe73c52cd56c600e55b75bc37ea69641bc0184b9ce9e76ca27311cf49566484f202df67d35558add045d580126876963832d7c5373584c34238f11e83bed4989bd77b85a30acaefdff5d88e61cbe790fe8be8bed5794e0a44f9a3e77525fcb825df714ab109654d60e5ef458a7744eaeb559b670c378bb8075fbaa8724e354a2c85811581b3d05d0fb08a24010d16ea9b0a807c2ff2c64315b5a5d01f7a26e51c540b8499ac948538d049c87a0fc6366a25385cc53d96320ca5d60ce14c4a0f71e581067d462a6c78bfaea139c1eb54f5202387607a741b489afbad4c37072fc99472198132ea694be9192ce4f5eeacb8a5c29a962bd855462f673fd4bce6dc8ffb5f3c59d58d022ee729e9f00e58c4110095f4bad44b76a1bdb9430f60718c3174c8fd7bc06acd9e245156df6da231badb2e2453163605d7d0524550a43fc56abb1fcb4b561c32264b100227a6635c029ffbb8033430795501d2b53d0a99fe7818f4246d3909c633a5e66a5d4b14c984c97f1e3fe0b347a5ccab21490628be0638c6a735dbbbf052df54fde4ce2ef0bde238c55d9e46f7468b5bf7f929548a717507a259a0cb571051ec4b52a22a781a64a95c00bf641eb3e855d7c1d6b5ca665847d43d3a9050e8a80cb12b598fa626db5130bdd683147a4e391f37a0c514317d078b551ba374384c46fdd366d7216fb0b0e1e9763c8f0637765ccf29fafe8f3db75635968d47aba91b5fef5c8970518f58fe1810e8c4e43e8e0efcb4dfcfcc34e7b3536da631af590a75d3ad2f0988c1b9c3ce29207c90bdf0a454d3dc2e74dd093132e3a6c8bfd694ce34db205351e2440170416c1ca8503ecb4c29888105a0ac4e58bad77349ac3ac7675915e4698b9805a2dc341b777fd8d257085d7e1b1a41693935d0d7f9446d0fffc978431c08166eda6c10ffaaa74f681a02d36c11c4591058f72b5d3b29bbf6728ebfe605f25f9b8b737b3f9e6112dae9bb72d1861953e23c614f898bd6a819442b654393c4c41a3a74db51ef27c3c96a64c444f493e24d682695d377e5a1f70911f0f26219f4797c02d9ed0985c2a1f0c1404cc9ce54b04099c6c16ba14a0e25f4fb68dd4c5127a48fcf6769e5900331336f4f82235bc96a5d3ad9e3c400cd1f26ed1d81e67aa3688e5437acd1cc5b905db9d3ba2d37e759b36a937cb856c2794a3e86ec72d18b1c80e0f43ed4d8199282a85fee66a998736efe74a2e8bc7cb8a1684b797be63d5509fe139c6943dbbc54c36a7b4a016b6bd23ef3d2efb25a51ff02c1b56c3938867c142c41f970007236838a0d51dfce977fd4a7a937d8e9d9a596902cbe3b1b4569bb94cc201d2658ca7114f118a5308ada27f6725b80b9b4278c2d4bfc7c4406657ce397578bbbd46314b1e2f8f24faa83d166eab55a3876176bff9fa17d414dc1a09f301a4c4f79a2d6d74f9d376a9adf498f6effb5f493d7ae8da447aea18030ff3e22e2ff4fbd0c300b835ff51fc07ef8c66b75345c91eadc6dd65980c34225b136fc3be54b1301d8b54acbabddbfc577a28cc0560f04e37e06b241a45508029f1008afcea01026096af26adc32ed654eaa3f6b8b5a4143fbeed4aa89a54669f16ff498ec0d48db7aa316b806adddb0d653ffc144eceb7ede9c99fe915281ce509d2c91fde101a484e4242dabbd154ab7264a8c33257806ae869eb86bdb4ea2452e3d2554a9ff13b0a4a7463bcf088559a7ac06a6925b7de4333f6269ae118847a982cd80e675ab966b1cac47723c13464e90db4fc22a47183918872b595ed2c085b9e5132529deaebdf09a5f4534570581a5746f2348e492fea681215b9b2458493f168bc4701f5203dfcad3a665e30fe7366cb5345bf91cfa55fadc236b0f677a7edc4b62dcf028d316a855146519cdf5d01e9273c7d6fec2c35ebc5f22c6868015a0e27ee503bbb241ad541d969de8241f2487701110b0de6b27625d771c2df0800ed1871c9c6317fd5441209ed1b6f85ff914a7cc38eba3b9df3eaccc62d6424a70a909e85cb77a8637acdbc247e2f93f16ca995da207b93c00d1db872d180bd160c975dcb7316098c207adc572c49d18f809b34a0e8d956e16ac585bed5ba3b169c3596e2e5950818de1f50cf5045a9a37628415b0f5becc9b2ea1af2629b076f627331134a7c0fce5bc600d110ac82f3f558a454bc8abe408e945f569faa87e9055816a6ca5485774076d4a89405c349a63445d318ec320ea6cc3c31b56f244b2eed2ee38ca4d2db3438b49b05d8d66b31c30c1f4b0bb1faa19999cdcb72465f84395f11b9f45dc2e01388f18bae22213d0ee2827c6fa4cd27045e700a3c56ebb044927a78cd90f4eb0f38f34d0d0f35721bad42d49e714fe21473639a54bab85916b064f2b5d238fe34ace97480056d7af457a87a22824877d1c0eea242dc8e28b89f184f368c58c18bcd1b278c6c93faa1c6fc96833d8091432e584a8f2335c53149deb6d8241adb1b81d08fbbac6d9d28e48652470cbfd673a3ebbcc2f674d78994fd4f220347f5ecb151d4942dc40893978dd4f856ccee6808ed1d63bf1406c86abf439c91a7b60746e71d9f585fe6eac95b441d8270822a46a798fae5defcf7b26abace3601c77326d892659308dd336136cb3ffa733ac243615210d0975e1e5447db415b1ecf383b6c5d6d47859578476791e5259dbd37e963762ce70ad743379ecf7c66c7775f6beda33fdd9fb8c06336bd976bb5b4911ff42ad794fe0d7045b7be07c1c267efa0a18f64bb1eeb149dea77de3485c6d9cc5d96983f3aa8ac65d9cf6bd2d746f0c1d246634bcad87bddcde940e8d39c16dae7e4f74e8751007f4e2a2f03d0bd6bb6c6f3c4f46b27fd0b253847ef68dd5030d7397c2aadb0727c5cf8dbbb74736667c5bc101dc880a20f3b8f7bb20c328fb678c0b03b746c0d424c61f8618a11ecbc19c9ce30aa8973d1571463978643dc54119e1ae9a42dd6ba668c0d5fdb3e029d09438694d9ed7739c1ef8707a0b73ef2e9fe439d049dd7ccf1a2a21c2fa3b2f2dc52c1b11d52b630c0496f6b2037be784906b1bd5d62176137ca3f004de703a3db1d05ef3e61399c9d4aa429cf6a63ce79deeeb377249e5f7e2d6d73ffb3fe984ecdddb603c10767a119adab75f935ddc1af82e51a8c75b00ff2b3dd0bafa9b97d89ce6e351a287711df1257da8b3c8b69a1f3bd0e41a33a32429aa63a820eefaf7a627f9c16a5b638c35c97c5c981c1b8dbcba11aec30e72e45a93654d535202dc82051005b850a5dee12d2005a0b68f78691fb0835259d4d2268f6214f2435409a904c2991d05dc787bc6f51e8e2666fc9ce329ec38946d5025f4c6999e5172e183bd9df618f96bfb7e2d881180fa565cd09aea53f64c90f0c17a1f35a80ac3f6ca85ee973c99d8da60d745894990b24b9cad7e450be0e4369175e883bfbdebdbb5f45106e865a797bc4ab9d048882f3b69a15259fa0fdb940e7e9f0e46094ee30e9f41cfaceb5cb5f90e51a0fe5f119ecffd02ed4117eb8ba10acf3fcb7b61cf0cdd5d5c0aa96ca79f88a955eb73fdf828370c8961a7989ff190d582c062b8d26934e026821e5bff28402d79906568da1ae5dbb8dc2d675368a76e91b57dd89f8c4c96d9a3c8af2f1bcb6a867da1d6b7007ffcf62117dc288ec4e6c71b1b35a8ee7addcfd026dab9f290ccf703c7cc01816bfccbd0f56b81cc5fdaad81d954711846218397b763223526d165d335bd6c063037945c54aa11ba0eb6bf6a16b97a92450e9f7af8ecea6bf4243d6c8a2f39177349fa33a3fc3ac02807773481d937d4edf16dfa79bc950f9f11f82023960b3d9bb067f761e68f8752495f8179483b68a7769986a86694887e4720ae909654d11fe767f82d0d626803f8f2d18b6af7b6080a8cb0038f7b7f181041585f84987a867b74e653c555fac392bb9526c5ef559629c1f7298bac4c0329c4cdfef568a255d909227492fcb7f7a3245cd9fd94cfc850d3dc78dfc987b5e0457fed7d27e1047cbf4d1e1aa91836a6b78923569a7d151c3cc0701035aaa6c02ddea96068bf28f2e1b6aeb10da4071a0fba818542e612f85753adb1933d6c65f86ac163ad7519c82b0fa4ea7cd3df89775172ee20b33851ce06903844b86d3e2fc4b3db68db8d68e5a8af4b8930265198fe8615d63d0bc51b49c608cc942fb55e4cff5c1d3aca51cc220c48a531722eef122970b61be3dea99d9dc7bd692a67ee94125a4d1b771f28fc7a667360595de42cdba362a876faf392a5e9799ff3c27b1a8b5a8b160a2d96a848295645328b12af2b124e9b2b96a3996cf2a3b102516e40c8c12982fa099fb61405d0489422e5397641dc4c885e14425157ae34eca142b75942db0add4fae965c9073e2f4a1f62c7bb6be3503add00e019d0ba3f91e82eecb4402161a4c4dd9803db3cceaa4a02879e556fd23f2615cec373e86de35c9478d49213d479be7e03d72123edc977d9e1f52290d7fe842cf8fcad946029360522dc6cad8cbcb275ad2d201e150f708fdca81b6de6debfed64aac4e69ec93f2bf1b5161713d366699fee2152cde4fccd809f882f87479eff1cb60107b12d3fabed45d87af128a203df006cea9db0e23bd6c3b1fd42a1eb0b2809ac1f21bbfe301e75fafeac2130eb774d490c0ddbeb80e5fadd00a0556cb88e8a0d5574dd43608d2ec3cace28d7c86ebc00e1a6cddf44b62d556028c9eaaf58206bd234fad7e341567d07dfa45e42cd34d890efef86314f45624463662c978eb510bee879836242d66edc65fd8e029b9f1a4171a7a7edba284ad4d794fc193630424019c40d859997de72ea9a2fb0e0e40f1eb25e0c20ffc0553a0ebc20dd5a784a804e423df34d8017f0f5e509f4a5ebfc4c3c7d7b7324f3cd9b9044618a017770a70f5bdd556678607f0f45e374f19c2cf6976abd8d33abf822e39687c83414fbdde577b352ed690047be2286dbccdfe642498f9a1833cf20f491185a41fa37354956fe8993e93a406f8de5bbcf3e9c7158dc53d9b92d478b3c2234a29a5bc86414ec5825bf62d3682001f448f850bb1661d81d441a28fb81201c51267e5fb0d1652fee4f7f601cc081a4b8f50009241d85c357a1e32cd34cc809d6af930fd5c465a626d17becfd8fbbfd998c8c72bfceeb2768b9bc42b7335d702d05817220843cd969725a0d5e1ff59400d347917570bc8bb8544b708f7d07dabdf0f190795b115020f9c5cc27b658f25394d6702c4d1f78af3a3bc04373edceee388b72c98f0c94b12f632f409d0c9efb95e417cdab3d75b532323abb5d38d428a4f61ff669572c86dcdd41911c3820619f178c41d33e0a3205d8e5df830f74d3fddb9fea3117a005c8da07027a0401ec7917d96e0d49bf8e7bd7c9982c9b37570be1fb8b2f777998ba038b7c0199c803ed8d063ea1861d315cd4eecca104a41d96b13ac3ab9d3ea67b00d4e5f71ffd10c1211998d61eab83707a5d642b71448b66f26479ecb17797d478ea27a5bd297abe7a1c63f539f1ed7f6e1d49100a9d0a424d82953bdcacd28a13d628539cd3a9ac41168946fe78d5505b366198f93903f46a9bfee04566c1687a9fd249e48305531d10f11196c7a4ace7b1960d63a0dc42f0b79e23162ccb35554d0bcebcc1f67ab711839e2967c35bba95cfb57358fff588de5c4b748758b84be9bd2b4c16e4ebb6212734dede98257deaf4da4eebd2d5d91920a289fed0ba0579cfa9bd09376cc348c51dddb68dd2d8b8ecc21fe1bfa98c6fd26e7d199c032b08298e372357c6868f459f4f7609297208a1b50cc02c3b48e95bf87860225b649bb322e1eaacc1a6fd8f035ca774f3f4e1fafd3d128351b68ea188a15ccb8a583c55afe4e89f24cc9b8c912bd035f8054cdfe1db5e2bb47d668d662094ff57b0f2f77b68a596b99f702290e579b6004e5038a80e2d8c2c467907d91e90278c14c681cf94a8fa46c8843a4ab12b5ed377f5ed9bfd060e80b9d0f82ac3d9a9ef2bbedbf390793494eedfbd05b86c95382743e702d171197a7d7246a236f6ef499ee5ed2f85a0aa6bba8a910a21c7c3f906430c25b1f615e03b67db87a06f5f611440a0cb9ba5c692d48ca0a115c150dcbdfc454bc61e6a7b9ab89d2128cd630136d9da0ef911efd3e3edd011bce4a5aaf4767bbd38708013436c76baf79762efacb78442c86c8cf48fb50ba43db7282a52ba94c42e75c070dd8b4d983455c4f41de287ad2f3c07be82ed38a0c1c2422680cf7aa6d0b39059111e09040890f0d83263c997570bc7f9619d3f4e6f8e5b6426bd13ab47b45997872b94469bce50639241344a6760346dd7cb08f670a2dcaeeb8d491dbb0084f3e6bdd5a6149a4b5de584af0d88c4b6e18a817ae0eb862bca77460242531d7732d7d9b46c9b79498dd5352127e2beb51f22c67f5309c3b28073f0286438959c1f018919df5af5b2bc085a74e19cd699ed48a3fe99f2e0d89a8794a2c992f05a810dd4ec487695efb226fe8f83df94ca14a41ee0f8cd6504de5a3958b988b4181fa2c315ef1ae16637d8265a55d281b15cdd420bb9c2dba01695e7327e426b40c2695c7d1c6c628013c6dcbd9e9e24b48c9b146823fbc91308dfa6185fe99c393209a90a17949fa50d9727a0a86a053c6e889edf809db7c57de17ff2a3a5d06b372bb2a527bb213eeff310ebb95fadb2c84ae75916d32d1e3c4379a8939d8e3e99777e3796096eb17f849d861aa5a37c6389f7bc2fc3b4860fac9d2277fa5e1a1f9415a6aaa5106886efd85913636f0407eed7f21195d365bcc4f24997e3c867932b7fdcacb88fc9cbdfc94debc19ab07b208ee1047213230c3480033a81558dbbe3c4bfe369713e18fa7748ece8abea89c824735282fb28c7a44f39a9fc8cf45f65ed3999e1402e33e8ea335c002a19f78ea28e41ec13dcb5bb68d42d7bf6044959023114478301e5522af78430a0a48e190f1c1bd9e3a89d22d2ba0c2619264065098e9ec4c31ef6e0407aaca194f5161273042670f4f2498195d58e267fc00b340c3a7d8087ef2e3933a61c7b8e735720d62eabdacd237f7fba0bfda3b03c736c121cf9a257db55060b621be5168619ec4182f13ef6a408c447469fc4330040b6241c5c53bb4a32eed095f207b596231ebbe842c5f8dcacf9476dbeb28f763ab94922ecba3ca48e9a8b29ea0af20b3e5d4d48e4965a3232c6bf68a0954f04281e"))
                    (acl2::hexstring=>ubyte8s "338a39c789636826a396214578166eb987b5f0b91ccdccf363ec13ad2262ce869fad36cd3eb7f89fea6b8135e2b2b24b9b0a2be7e57db1326c370dd7424e2a85"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "cbee0413f1f169a6512ecaefcf5a0c665ddbc5f5365749da16dd98a04b9939584004295aae3867f2697c52a28be0e42db6d1d20954e386dbdeec744770fdd2ba"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "ec4ae4e3668035491074838888491390f18c964bbf2aa16f44f76a167ea90043cbdce5fe79fe09c2e78cdb1fbfb8c3eab143299bb7ef98f9787a4a9f94ca413e"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "fa933b224e0912687b04a51ddbe1383a5c1ccfd2c0c15a3aa8b40844fe70f356afa3a3317175b664cd3e01854adfb1e2a6e20fb2a4720d53c4fa9ba555dd5e9d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "d4292e3e757b4819873b554278a8c484e99c019a18d1073459c56ac454c2d518298050a78e46e2af85b25e05fcff6ad5fed4c436577a90a3107f25b99f1cc999"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "06b3fe00a2c681dbc14f6e81ae1e7addedeeaaf4c7485d28beed0b52d128cb9e7b295838dad384054a826cd316fe20b8ea1aa35de2b39ae612342dd1911212d9"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "8bcf4c7df9b79044ab522ecb2d9bb9dcf54b6e131404f1f2bb413b2790968c2b83c4d34c61edef8302e3129e67e3f95b249d160ae5e0c4d3488079263e5ed128"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "c9c7d9b0413dd75496646779efd98d505921b9a547cbd5c460450b24fcb8119a769c27a9b63e300402cf04442e900cd496c0a05682f88c820d5a6d431005eff8"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "5169e69bf0e777581ca94e070b47454ad7f6af034dd3e2ed564d020f6557fd908f721e2b19e548fb0c41763fe0abdd89efdacce8e3546faf22f3fd089928b019"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "68c34d5aec130f666c98d7088506e213c84ff3a7e86c21932039242a68b4b78e36b7b3948435d89d2a86a5194ba1b321832ae7b8ec286f86b6dbe3e39da2d2a1"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "9dce12e49fae09e0b1a60f44199449fa31b9bb277857a84efb874b0c23118bf8453f8435f47b161c9747e719a642f805bf14c06b6202535e7abacd53e25ab84f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "e95fda82d0384a00d1aaf90744ec582cdbbe31f5c6b2ea26b963ff633c7224340fc4423203a8972fe7d90c2a2342cdb8fbe73e0ab036a29b81d08d859978281f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "a4245a5f76f6228113344929db4ea36401f1b48ecde3714ea7e9f90132f08658457e3b90d935c5e400da23671ab42ca836f0b070609d4dd94480b24531026eff"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "2529d5d0f4625ae701397a41b5184bbd45b840c3fee6652a467ff1a49c622604991157b16393136cd123a515e4a0412dd2bf2b3227dffb652e11a6304523c31d"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "c8b144ba9b9a8ae0ff3cf8a6444cf9291cb72feb3d8402237d2598a57d0d8389b198f67b5203576fa8f878ef2728644f706421fbc3db966b967cad4990a49480"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "bcf7e3ac677b340318080a6212387096153ed60c24b1ac5c1744575e7f94fd0363d2e0ca6fa4d7445924963aafb47cec33632399c3533bbc144bbaffeeabe46f"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "aa13b83b4db9a58a7684bad6b864abdad23bdd71b4e5bcab1a8750aa8e6b1ca70730a7b9ff684fa6b73a70ca75c55aee6afe1b6f573c8fb6ecf3962a30e0af0b"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "6730a70ed95c48d1fa1c04dd32c192337ab47f4df3c6a1e9d101c0983cceded1a10ab25fe0437de24a6f9e3848f892c2327fc7fa6a0fb1854d81bf05d9733573"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "c49a67a96dccfe9597d5c1070077f46024b331f4b64d8713709f9ab3248ec7b95319da1ac43e0d67e2f3ac558943a5fd9ed2a67decbe6c873efe01f59db57cb8"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "abdeca34f0dfd5d9097f0ea9e44263d17d93c6e02372c2e8896e8bd8c32957ea39055a956e4f803ea2a341166a87edb91cf0adb758fd39eae9438d0b2cb2cd39"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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"))
                    (acl2::hexstring=>ubyte8s "956d5ca8fb9c84cdc882eb0bf3f5ff0a61238f982286673bb9185a45858ea13827dee168b4abf313a0b82aa5e98d2a01510dfc9e2228c657425f2941ff0c6d63"))

(acl2::assert-equal (sha-512-bytes (acl2::hexstring=>ubyte8s "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