; Grouping and Ungrouping of Digits in Specific Bases
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(include-book "defdigits")

(include-book "xdoc/defxdoc-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc defdigit-grouping

  :parents (digits-any-base)

  :short
  (xdoc::topstring
   "Generate specialized versions of "
   (xdoc::seetopic
    "digits-any-base"
    "the operations to group and ungroup digits in arbitrary bases")
   ", based on existing instances of @(tsee defdigits).")

  :long

  (xdoc::topstring

   (xdoc::h3 "Introduction")

   (xdoc::p
    "The @(tsee defdigits) macro generates specialized versions of
     the operations to convert between natural numbers and digits.
     This macro, @('defdigit-grouping'), generates specialized versions of
     the operations to convert
     between digits in smaller bases and digits in larger bases.")

   (xdoc::p
    "The @('defdigit-grouping') macro references
     the names of two existing calls of @(tsee defdigits),
     which determine the smaller and larger bases,
     as well as the specialized conversions to use
     to define the grouping and ungrouping functions
     generated by @('defdigit-grouping').")

   (xdoc::h3 "General Form")

   (xdoc::codeblock
    "(defdigit-grouping name"
    "                   :smaller ..."
    "                   :larger ..."
    "                   :group-bendian ..."
    "                   :group-lendian ..."
    "                   :ungroup-bendian ..."
    "                   :ungroup-lendian ..."
    "                   :parents ..."
    "                   :short ..."
    "                   :long ..."
    "  )")

   (xdoc::h3 "Inputs")

   (xdoc::desc
    "@('name')"
    (xdoc::p
     "A symbol that names the ensemble of
      the generated specialized grouping and ungrouping functions.
      This is used as the XDOC topic under which
      the XDOC topics for the generated events are put."))

   (xdoc::desc
    "@(':smaller')"
    (xdoc::p
     "The name of a previous successful call of @(tsee defdigits).
      This determines the smaller base for grouping and ungrouping.
      Let @('base') be the smaller base."))

   (xdoc::desc
    "@(':larger')"
    (xdoc::p
     "The name of a previous successful call of @(tsee defdigits).
      This determines the larger base for grouping and ungrouping,
      which must be a positive power of @('base') that is at least 2.
      Let @('exp') be that positive power,
      so that the larger base is @('base^exp')."))

   (xdoc::desc
    (list
     "@(':group-bendian')"
     "@(':group-lendian')"
     "@(':ungroup-bendian')"
     "@(':ungroup-lendian')")
    (xdoc::p
     "Symbols that specify the names to use for the generated functions
      (see details below)."))

   (xdoc::desc
    (list
     "@(':parents')"
     "@(':short')"
     "@(':long')")
    (xdoc::p
     "These, if present, are added to the XDOC topic generated for
      the ensemble of
      the generated specialized grouping and ungrouping functions."))

   (xdoc::h3 "Generated Events")

   (xdoc::desc
    (list
     "@('group-bendian')"
     "@('group-lendian')"
     "@('ungroup-bendian')"
     "@('ungroup-lendian')")
    (xdoc::p
     "Specialized versions of
      @(tsee group-bendian),
      @(tsee group-lendian),
      @(tsee ungroup-bendian), and
      @(tsee ungroup-lendian).
      These new functions are accompanied by theorems corresponding to
      the ones that accompany the original functions."))

   (xdoc::p
    "The generated events include XDOC documentation.
     They are all under an XDOC for the ensemble,
     whose name is specified in the @('name') input.")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ defdigit-grouping-implementation
  :parents (defdigit-grouping)
  :short "Implementation of @(tsee defdigit-grouping)."
  :order-subtopics t
  :default-parent t)

(define defdigit-grouping-find-exp ((smaller dab-basep) (larger dab-basep))
  :returns (exp? maybe-posp)
  :verify-guards nil ; done below
  :short "Find the exponent @('exp')
          that relates the smaller and larger bases."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the logarithm to base @('smaller') of @('larger'),
     i.e. the exponent @('exp') such that @('larger') is @('smaller^exp').
     If @('exp') is not a positive integer, we return @('nil').
     If the two bases are the same, we return 1."))
  (b* (((unless (mbt (and (dab-basep smaller)
                          (dab-basep larger))))
        nil) ; never happens
       (larger2 (/ larger smaller))
       ((unless (posp larger2)) nil)
       ((when (= larger2 1)) 1)
       (exp2? (defdigit-grouping-find-exp smaller larger2))
       ((unless exp2?) nil))
    (1+ exp2?))
  :prepwork ((local (include-book "arithmetic/top-with-meta" :dir :system)))
  ///
  (verify-guards defdigit-grouping-find-exp
    :hints (("Goal" :in-theory (enable dab-basep)))))

(define defdigit-grouping-fn (name
                              smaller
                              larger
                              group-bendian
                              group-lendian
                              ungroup-bendian
                              ungroup-lendian
                              parents
                              short
                              long
                              (wrld plist-worldp))
  :returns (event "A @(tsee pseudo-event-formp).")
  :mode :program
  :short "Event generated by @(tsee defdigit-grouping)."
  (b* (;; validate the NAME input:
       ((unless (symbolp name))
        (raise "The NAME input must be a symbol, ~
                 but it is ~x0 instead." name))
       ;; validate the :SMALLER input:
       ((unless (symbolp smaller))
        (raise "The :SMALLER input must be a symbol, ~
                 but it is ~x0 instead." smaller))
       (pair (assoc-eq smaller (table-alist *defdigits-table-name* wrld)))
       ((unless pair)
        (raise "The :SMALLER input ~x0 must be the name of ~
                an existing DEFDIGITS, but it is not." smaller))
       ((defdigits-info smaller) (cdr pair))
       ;; validate the :LARGER input:
       ((unless (symbolp larger))
        (raise "The :LARGER input must be a symbol, ~
                 but it is ~x0 instead." larger))
       (pair (assoc-eq larger (table-alist *defdigits-table-name* wrld)))
       ((unless pair)
        (raise "The :LARGER input ~x0 must be the name of ~
                an existing DEFDIGITS, but it is not." larger))
       ((defdigits-info larger) (cdr pair))
       ;; ensure that the larger base is a positive power of the smaller base,
       ;; where the positive power is more than 1,
       ;; and find the exponent that relates the two bases:
       (exp? (defdigit-grouping-find-exp smaller.base larger.base))
       ((unless exp?)
        (raise "The :LARGER input ~x0 must be a positive power of ~
                the :SMALLER input ~x1, but it is not." larger smaller))
       (exp exp?)
       ((when (= exp 1))
        (raise "The :LARGER input ~x0 must be a positive power of ~
                the :SMALLER input ~x1 and must be larger, ~
                but this is not the case." larger smaller))
       ;; validate the GROUP-BENDIAN input:
       ((unless (symbolp group-bendian))
        (raise "The :GROUP-BENDIAN input must be a symbol, ~
                 but it is ~x0 instead." group-bendian))
       ;; validate the GROUP-LENDIAN input:
       ((unless (symbolp group-lendian))
        (raise "The :GROUP-LENDIAN input must be a symbol, ~
                 but it is ~x0 instead." group-lendian))
       ;; validate the UNGROUP-BENDIAN input:
       ((unless (symbolp ungroup-bendian))
        (raise "The :UNGROUP-BENDIAN input must be a symbol, ~
                 but it is ~x0 instead." ungroup-bendian))
       ;; validate the UNGROUP-LENDIAN input:
       ((unless (symbolp ungroup-lendian))
        (raise "The :UNGROUP-LENDIAN input must be a symbol, ~
                 but it is ~x0 instead." ungroup-lendian))
       ;; names of the generated theorems:
       (len-of-group-bendian (packn-pos (list 'len-of- group-bendian)
                                        name))
       (len-of-group-lendian (packn-pos (list 'len-of- group-lendian)
                                        name))
       (len-of-ungroup-bendian (packn-pos (list 'len-of- ungroup-bendian)
                                          name))
       (len-of-ungroup-lendian (packn-pos (list 'len-of- ungroup-lendian)
                                          name))
       ;; names of the variables used in the generated events:
       (digits (packn-pos (list "DIGITS") name))
       (new-digits (packn-pos (list "NEW-DIGITS") name))
       ;; generated events:
       (group-bendian-event
        `(define ,group-bendian ((,digits ,smaller.digits-pred))
           :guard (integerp (/ (len ,digits) ,exp))
           :returns (,new-digits
                     ,larger.digits-pred
                     :hints (("Goal"
                              :in-theory '(,group-bendian
                                           ,larger.digits-pred-correct
                                           return-type-of-group-bendian
                                           (:e dab-base-fix)
                                           (:e pos-fix)
                                           (:e expt)))))
           :parents (,name)
           :short (xdoc::topstring
                   "Group "
                   ,smaller.digits-description
                   " into "
                   ,larger.digits-description
                   ", big-endian.")
           (group-bendian ,smaller.base ,exp ,digits)
           :guard-hints (("Goal" :in-theory '(,smaller.digits-pred-correct
                                              (:e dab-basep)
                                              (:e posp)
                                              (:e pos-fix))))
           ///
           (fty::deffixequiv ,group-bendian
             :hints (("Goal"
                      :in-theory
                      '(,smaller.digits-fix-correct
                        ,group-bendian
                        group-bendian-of-dab-digit-list-fix-digits))))))
       (group-lendian-event
        `(define ,group-lendian ((,digits ,smaller.digits-pred))
           :guard (integerp (/ (len ,digits) ,exp))
           :returns (,new-digits
                     ,larger.digits-pred
                     :hints (("Goal"
                              :in-theory '(,group-lendian
                                           ,larger.digits-pred-correct
                                           return-type-of-group-lendian
                                           (:e dab-base-fix)
                                           (:e pos-fix)
                                           (:e expt)))))
           :parents (,name)
           :short (xdoc::topstring
                   "Group "
                   ,smaller.digits-description
                   " into "
                   ,larger.digits-description
                   ", little-endian.")
           (group-lendian ,smaller.base ,exp ,digits)
           :guard-hints (("Goal" :in-theory '(,smaller.digits-pred-correct
                                              (:e dab-basep)
                                              (:e posp)
                                              (:e pos-fix))))
           ///
           (fty::deffixequiv ,group-lendian
             :hints (("Goal"
                      :in-theory
                      '(,smaller.digits-fix-correct
                        ,group-lendian
                        group-lendian-of-dab-digit-list-fix-digits))))))
       (ungroup-bendian-event
        `(define ,ungroup-bendian ((,digits ,larger.digits-pred))
           :returns (,new-digits
                     ,smaller.digits-pred
                     :hints (("Goal"
                              :in-theory '(,ungroup-bendian
                                           ,smaller.digits-pred-correct
                                           return-type-of-ungroup-bendian))))
           :parents (,name)
           :short (xdoc::topstring
                   "Ungroup "
                   ,larger.digits-description
                   " into "
                   ,smaller.digits-description
                   ", big-endian.")
           (ungroup-bendian ,smaller.base ,exp ,digits)
           :guard-hints (("Goal" :in-theory '(,larger.digits-pred-correct
                                              (:e dab-basep)
                                              (:e dab-base-fix)
                                              (:e posp)
                                              (:e pos-fix)
                                              (:e expt))))
           ///
           (fty::deffixequiv ,ungroup-bendian
             :hints (("Goal"
                      :in-theory
                      '(,larger.digits-fix-correct
                        ,ungroup-bendian
                        ungroup-bendian-of-dab-digit-list-fix-digits
                        (:e dab-base-fix)
                        (:e pos-fix)
                        (:e expt)))))))
       (ungroup-lendian-event
        `(define ,ungroup-lendian ((,digits ,larger.digits-pred))
           :returns (,new-digits
                     ,smaller.digits-pred
                     :hints (("Goal"
                              :in-theory '(,ungroup-lendian
                                           ,smaller.digits-pred-correct
                                           return-type-of-ungroup-lendian))))
           :parents (,name)
           :short (xdoc::topstring
                   "Ungroup "
                   ,larger.digits-description
                   " into "
                   ,smaller.digits-description
                   ", little-endian.")
           (ungroup-lendian ,smaller.base ,exp ,digits)
           :guard-hints (("Goal" :in-theory '(,larger.digits-pred-correct
                                              (:e dab-basep)
                                              (:e dab-base-fix)
                                              (:e posp)
                                              (:e pos-fix)
                                              (:e expt))))
           ///
           (fty::deffixequiv ,ungroup-lendian
             :hints (("Goal"
                      :in-theory
                      '(,larger.digits-fix-correct
                        ,ungroup-lendian
                        ungroup-lendian-of-dab-digit-list-fix-digits
                        (:e dab-base-fix)
                        (:e pos-fix)
                        (:e expt)))))))
       (len-of-group-bendian-event
        `(defrule ,len-of-group-bendian
           (equal (len (,group-bendian ,digits))
                  (ceiling (len ,digits) ,exp))
           :in-theory '(,group-bendian
                        len-of-group-bendian
                        (:e pos-fix))))
       (len-of-group-lendian-event
        `(defrule ,len-of-group-lendian
           (equal (len (,group-lendian ,digits))
                  (ceiling (len ,digits) ,exp))
           :in-theory '(,group-lendian
                        len-of-group-lendian
                        (:e pos-fix))))
       (len-of-ungroup-bendian-event
        `(defrule ,len-of-ungroup-bendian
           (equal (len (,ungroup-bendian ,digits))
                  (* (len ,digits) ,exp))
           :in-theory '(,ungroup-bendian
                        len-of-ungroup-bendian
                        (:e pos-fix))))
       (len-of-ungroup-lendian-event
        `(defrule ,len-of-ungroup-lendian
           (equal (len (,ungroup-lendian ,digits))
                  (* (len ,digits) ,exp))
           :in-theory '(,ungroup-lendian
                        len-of-ungroup-lendian
                        (:e pos-fix))))
       (name-event
        `(defxdoc ,name
           ,@(and parents (list :parents parents))
           ,@(and short (list :short short))
           ,@(and long (list :long long)))))
    ;; top-level event:
    `(encapsulate
       ()
       (logic)
       (set-default-hints nil)
       (set-override-hints nil)
       ,group-bendian-event
       ,group-lendian-event
       ,ungroup-bendian-event
       ,ungroup-lendian-event
       ,len-of-group-bendian-event
       ,len-of-group-lendian-event
       ,len-of-ungroup-bendian-event
       ,len-of-ungroup-lendian-event
       ,name-event)))

(defsection defdigit-grouping-macro-definition
  :short "Definition of the @(tsee defdigit-grouping) macro."
  :long (xdoc::topstring-@def "defdigit-grouping")
  (defmacro defdigit-grouping (name
                               &key
                               smaller
                               larger
                               group-bendian
                               group-lendian
                               ungroup-bendian
                               ungroup-lendian
                               parents
                               short
                               long)
    `(make-event (defdigit-grouping-fn
                   ',name
                   ',smaller
                   ',larger
                   ',group-bendian
                   ',group-lendian
                   ',ungroup-bendian
                   ',ungroup-lendian
                   ',parents
                   ',short
                   ',long
                   (w state)))))
