/*
 * Copyright (C) 2002 Stefan Holst
 * Copyright (C) 2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * otk - the osd toolkit
 */

#ifndef HAVE_OTK_H
#define HAVE_OTK_H

#include "../xine-toolkit/dlist.h"
#include "odk.h"
#include "list.h"
#include "oxine_event.h"

#undef OTK_WITH_SLIDER

/** If a string starts with OTK_TEMP_STR, then otk will copy the rest of it (slower).
 *  Otherwise, otk will use the string as is, and you shall keep it valid until widget delete. */
#define OTK_TEMP_STR "\x01"

/** sized list helper. */
#define SIZED_LIST_SAMPLE_LD 4
typedef struct {
  xitk_dlist_t list;
  xitk_dnode_t *last_n[(1<<SIZED_LIST_SAMPLE_LD)+2];
  int last_i[(1<<SIZED_LIST_SAMPLE_LD)+2], num;
} sized_list_t;
void sized_list_init (sized_list_t *list);
xitk_dnode_t *sized_list_get (sized_list_t *list, int i);
void sized_list_insert (sized_list_t *list, xitk_dnode_t *item, int index);
void sized_list_free (sized_list_t *list, void (*node_free) (xitk_dnode_t *node));

/** alignments for text */
#define OTK_ALIGN_LEFT     0x001
#define OTK_ALIGN_CENTER   0x002
#define OTK_ALIGN_RIGHT    0x004
#define OTK_ALIGN_TOP      0x008
#define OTK_ALIGN_VCENTER  0x010
#define OTK_ALIGN_BOTTOM   0x020

/** opaque data types */
typedef struct otk_s otk_t;
typedef struct otk_widget_s otk_widget_t;

/** user contributed callbacks */
typedef enum {
  OTK_ACT_NULL = 0,
  OTK_ACT_HIT,
  OTK_ACT_STEP,
  OTK_ACT_VALUE,
  OTK_ACT_LAST
} otk_widget_action_t;
typedef void (*otk_widget_cb_t) (otk_widget_t *widget, void *user_data, int action, int value);
typedef void (*otk_button_cb_t) (void *user_data);
typedef void (*otk_list_cb_t) (void *user_data, void *entry);
#ifdef OTK_WITH_SLIDER
typedef int  (*otk_slider_cb_t) (odk_t *odk);
#endif

/** initialisation */
otk_t *otk_init (oxine_t *oxine, int verbosity);

/** creating widgets */
otk_widget_t *otk_window_new (otk_t *otk, const char *title, int x, int y,
			      int w, int h);
otk_widget_t *otk_button_new (otk_widget_t *win, int x, int y,
			      int w, int h, const char *text,
			      otk_button_cb_t cb,
			      void *user_data);
otk_widget_t *otk_button_grid_new (const char *text,
			      otk_button_cb_t cb,
			      void *user_data);
otk_widget_t *otk_label_new (otk_widget_t *win, int x, int y, int alignment,
                             const char *text);
otk_widget_t *otk_list_new (otk_widget_t *win, int x, int y, int w, int h,
			  otk_list_cb_t cb,
			  void *user_data);
#ifdef OTK_WITH_SLIDER
otk_widget_t *otk_slider_new (otk_widget_t *win, int x, int y, int w, int h, otk_slider_cb_t cb);
otk_widget_t *otk_slider_grid_new (otk_slider_cb_t cb);
#endif
otk_widget_t *otk_selector_new(otk_widget_t *win, int x, int y, int w, int h,
  const char *const *items, int num, otk_widget_cb_t user_cb, void *user_data);
otk_widget_t *otk_selector_grid_new (const char *const *items, int num,
  otk_widget_cb_t user_cb, void *cb_data);
otk_widget_t *otk_layout_new(otk_widget_t *win, int x, int y, int w, int h, int rows, int columns);
otk_widget_t *otk_scrollbar_new (otk_widget_t *win, int x, int y, int w, int h, otk_widget_cb_t user_cb, void *user_data);

/** geeric for all widgets. */
const char *otk_widget_text_get (otk_widget_t *widget);

/** scrollbar widget functions */
#define OTK_SCROLLBAR_FULL_LD 10
#define OTK_SCROLLBAR_FULL_VAL (1 << OTK_SCROLLBAR_FULL_LD)
void otk_scrollbar_set (otk_widget_t *w, int pos_start, int pos_end);

/** selector widget functions */
void otk_selector_set (otk_widget_t *w, int pos);

/** label widget functions */
void otk_label_set_font_size (otk_widget_t *w, int font_size);

/** button widget functions */
void otk_button_uc_set (otk_widget_t *w, otk_button_cb_t uc, void *data);

/** layout widget functions */
/* layout widget use widget's x,y,w,h to set positions and sizes :
 * x = start row : y = start column
 * w = columns to be uses : h = rows to be used */
void otk_layout_add_widget (otk_widget_t *layout, otk_widget_t *widget, int srow, int scol, int frow, int fcol);

/** list widget functions */
typedef enum {
  OTK_GET_PTR_ITEM = 0, /** << the address you want to see in the select callback. */
  OTK_GET_PTR_TEXT      /** << (const char *). */
} otk_get_ptr_what_t;
typedef void *(*otk_get_ptr_t)(xitk_dnode_t *item, otk_get_ptr_what_t what);
/** user_list == NULL to reset. */
void otk_set_user_list (otk_widget_t *list, sized_list_t *user_list, otk_get_ptr_t get_ptr);
void otk_set_user_array (otk_widget_t *list, const char * const *texts, int n, int mark, otk_widget_cb_t user_cb, void *user_data);
int otk_list_get_pos (otk_widget_t *w);
void otk_list_set_pos (otk_widget_t *w, int newpos);

/** update drawing area */
void otk_draw_all (otk_t *otk);

/** set focus to a specific widget (causes buttons lighting up) */
void otk_set_focus (otk_widget_t *w);

/** frees given widget and all its childs */
void otk_destroy_widget (otk_widget_t *w);

/** frees all widgets */
void otk_clear (otk_t *otk);

/** frees otk struct */
void otk_free (otk_t *otk);

/** event stuff */
void otk_set_event_handler (otk_t *otk, int (*cb)(void *data, oxine_event_t *ev), void *data);
int otk_send_event (otk_t *otk, oxine_event_t *event);

#endif
